# LarBot Template - Modular Telegram Bot Framework

[![Laravel](https://img.shields.io/badge/Laravel-10.x-red.svg)](https://laravel.com)
[![PHP](https://img.shields.io/badge/PHP-8.3+-blue.svg)](https://php.net)
[![Filament](https://img.shields.io/badge/Filament-3.2-orange.svg)](https://filamentphp.com)
[![License](https://img.shields.io/badge/License-MIT-green.svg)](LICENSE)

A **production-ready, modular Telegram bot framework** built on Laravel 10.x with comprehensive Persian language support. Originally derived from an Iranian automotive bot (HajiBot), this template has been completely transformed into a **generic, extensible foundation** for building any type of Telegram bot.

## 🚀 Key Features

### 🏗️ **Modular Architecture**
- **Pluggable Module System**: Easy-to-develop modules with standardized interfaces
- **Event-Driven Communication**: Modules communicate via EventBus for loose coupling
- **Configuration Management**: Layered configuration system with runtime updates
- **Hot-Swappable Components**: Enable/disable modules without code changes

### 🎛️ **Admin Panel (Filament v3.2)**
- **User Management**: Complete CRUD operations with role-based access
- **System Monitoring**: Real-time health checks and performance metrics
- **Settings Management**: Runtime configuration via elegant admin interface
- **Analytics Dashboard**: Comprehensive user behavior tracking and insights

### 📱 **Essential Modules**

#### 🇮🇷 **Persian Localization**
- Complete RTL text support with proper formatting
- Persian number conversion and date handling
- Cultural adaptations for Iranian users
- Flexible translation system

#### 💳 **Payment Integration**
- **Zarinpal Gateway**: Full integration for Iranian market
- Subscription management with automatic renewals
- Transaction tracking and reconciliation
- Multi-currency support with IRR focus

#### 📢 **Notification System**
- **Broadcast Campaigns**: Scheduled mass messaging with targeting
- **Template Management**: Reusable message templates with variables
- **User Preferences**: Granular opt-in/opt-out controls
- **Delivery Analytics**: Real-time campaign performance metrics

#### 📊 **Analytics & Insights**
- **User Behavior Tracking**: Comprehensive interaction logging
- **Performance Metrics**: Response times, success rates, usage patterns
- **Custom Events**: Track business-specific user actions
- **Data Export**: CSV/Excel exports for external analysis

#### 📝 **Content Management**
- **Dynamic Menus**: Admin-configurable bot menus and workflows
- **Content Pages**: Rich text content with media support
- **SEO-Friendly**: Optimized content structure
- **Version Control**: Content revision history

## 🛠️ Technology Stack

- **Backend**: Laravel 10.x with PHP 8.3+
- **Database**: MySQL 8.0+ with optimized indexing
- **Admin Panel**: Filament v3.2 with custom resources
- **Telegram API**: DefStudio/Telegraph for reliable bot communication
- **Queue System**: Database-driven job processing
- **Caching**: File-based caching with Redis support
- **Hosting**: Optimized for cPanel shared hosting

## 📦 Quick Start

### Prerequisites
- PHP 8.3+
- MySQL 8.0+
- Composer 2.x
- Node.js 18+ (for asset compilation)

### Installation

1. **Clone the repository**
   ```bash
   git clone https://github.com/your-username/larbot-template.git
   cd larbot-template
   ```

2. **Install dependencies**
   ```bash
   composer install
   npm install && npm run build
   ```

3. **Environment setup**
   ```bash
   cp .env.example .env
   php artisan key:generate
   ```

4. **Configure your .env file**
   ```env
   # Database
   DB_CONNECTION=mysql
   DB_HOST=127.0.0.1
   DB_PORT=3306
   DB_DATABASE=larbot_template
   DB_USERNAME=your_username
   DB_PASSWORD=your_password

   # Telegram Bot
   TELEGRAM_BOT_TOKEN=your_bot_token
   TELEGRAM_BOT_USERNAME=your_bot_username

   # Zarinpal (for Iranian market)
   ZARINPAL_MERCHANT_ID=your_merchant_id
   ZARINPAL_SANDBOX=true
   ```

5. **Database setup**
   ```bash
   php artisan migrate
   php artisan db:seed
   ```

6. **Setup Telegram webhook**
   ```bash
   php artisan telegram:setup
   ```

7. **Start the application**
   ```bash
   # Development
   php artisan serve
   
   # Production (cPanel)
   # Upload files and configure as per deployment guide
   ```

## 🏗️ Architecture Overview

### Module System
The framework uses a **pluggable module architecture** where each module implements the `TelegramModuleInterface`:

```php
interface TelegramModuleInterface
{
    public function getModuleName(): string;
    public function handleMessage(TelegramMessage $message): ?TelegramResponse;
    public function handleCallback(TelegramCallback $callback): ?TelegramResponse;
    public function getConfigurationSchema(): array;
    public function install(): void;
    public function uninstall(): void;
}
```

### Core Services

#### **ModuleManager**
Handles module registration, lifecycle, and routing:
```php
$moduleManager = app(ModuleManager::class);
$moduleManager->registerModule(new PaymentModule());
$moduleManager->handleMessage($telegramMessage);
```

#### **EventBus**
Facilitates inter-module communication:
```php
EventBus::emit('user.registered', ['user' => $user]);
EventBus::listen('payment.completed', [$this, 'handlePaymentCompleted']);
```

#### **ConfigurationManager**
Manages layered configuration with runtime updates:
```php
$config = app(ConfigurationManager::class);
$config->set('payment.zarinpal.sandbox_mode', false);
$apiKey = $config->get('telegram.bot_token');
```

## 📚 Available Modules

### 1. **Persian Localization Module**
- **Purpose**: Complete Persian language support
- **Commands**: `/language`, `/calendar`
- **Features**: RTL text, Persian dates, number conversion

### 2. **Payment Module**
- **Purpose**: Handle payments via Zarinpal
- **Commands**: `/payment`, `/subscription`, `/invoice`
- **Features**: Recurring billing, transaction history, refunds

### 3. **Notification Module**
- **Purpose**: Broadcast messaging system
- **Commands**: `/notifications`, `/subscribe`, `/unsubscribe`
- **Features**: Campaign management, user targeting, analytics

### 4. **Analytics Module**
- **Purpose**: User behavior tracking
- **Commands**: `/stats` (admin only)
- **Features**: Event tracking, performance metrics, reporting

### 5. **Content Module**
- **Purpose**: Dynamic content management
- **Commands**: `/content`, `/menu`, `/help`
- **Features**: Page builder, menu configuration, SEO optimization

## 🔧 Configuration

### Module Configuration
Each module can be configured via the admin panel or configuration files:

```php
// config/modules.php
return [
    'payment' => [
        'enabled' => true,
        'zarinpal' => [
            'merchant_id' => env('ZARINPAL_MERCHANT_ID'),
            'sandbox_mode' => env('ZARINPAL_SANDBOX', true),
        ],
    ],
    'notification' => [
        'enabled' => true,
        'rate_limit_hours' => 24,
        'default_opt_in' => true,
    ],
];
```

### cPanel Hosting Configuration
The template includes specific optimizations for cPanel hosting:

```php
// config/cpanel.php
return [
    'deployment' => [
        'public_html_symlink' => true,
        'storage_symlink' => true,
        'optimize_for_cpanel' => true,
    ],
    'queue' => [
        'driver' => 'database',
        'retry_after' => 90,
    ],
    'optimization' => [
        'config_cache' => true,
        'route_cache' => true,
        'view_cache' => true,
    ],
];
```

## 🚀 Deployment

### cPanel Deployment
1. **Upload files** to your hosting account
2. **Configure database** via cPanel MySQL interface  
3. **Set up cron jobs** for queue processing:
   ```bash
   */5 * * * * cd /path/to/larbot && php artisan schedule:run
   ```
4. **Configure webhooks** via admin panel
5. **Test bot functionality** using the built-in testing tools

For detailed deployment instructions, see [DEPLOYMENT.md](docs/DEPLOYMENT.md).

## 📖 Documentation

- **[Installation Guide](docs/INSTALLATION.md)** - Detailed setup instructions
- **[Module Development](docs/MODULE_DEVELOPMENT.md)** - Creating custom modules
- **[API Reference](docs/API.md)** - Complete API documentation
- **[Deployment Guide](docs/DEPLOYMENT.md)** - Production deployment
- **[Configuration Reference](docs/CONFIGURATION.md)** - All configuration options
- **[Troubleshooting](docs/TROUBLESHOOTING.md)** - Common issues and solutions

## 🧪 Testing

The framework includes comprehensive testing tools:

```bash
# Run all tests
php artisan test

# Run specific test suites
php artisan test --testsuite=Feature
php artisan test --testsuite=Unit

# Generate test coverage report
php artisan test --coverage
```

## 🛡️ Security

- **Input Validation**: All user inputs are validated and sanitized
- **CSRF Protection**: Built-in Laravel CSRF protection
- **Rate Limiting**: Configurable rate limiting per user/IP
- **SQL Injection Prevention**: Eloquent ORM with parameterized queries
- **XSS Protection**: Automatic escaping of user content

## 🤝 Contributing

We welcome contributions! Please see [CONTRIBUTING.md](CONTRIBUTING.md) for guidelines.

1. Fork the repository
2. Create a feature branch
3. Make your changes
4. Add tests for new functionality
5. Submit a pull request

## 📄 License

This project is licensed under the MIT License - see [LICENSE](LICENSE) file for details.

## 🆘 Support

- **Documentation**: Comprehensive guides in the `/docs` directory
- **Issues**: Report bugs via GitHub Issues
- **Community**: Join our discussions in GitHub Discussions
- **Commercial Support**: Available for enterprise deployments

## 🙏 Acknowledgments

- **Laravel Framework** - Robust PHP framework foundation
- **Filament** - Beautiful admin panel solution
- **DefStudio Telegraph** - Reliable Telegram Bot API wrapper
- **Iranian Developer Community** - Inspiration and feedback

---

**Built with ❤️ for the Iranian developer community and anyone building Telegram bots worldwide.**