<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class CleanupAdsData extends Command
{
    protected $signature = 'ads:cleanup {--days=7 : Days to keep data} {--dry-run : Show what would be deleted}';
    protected $description = 'Clean up old ads data';

    public function handle()
    {
        $days = (int) $this->option('days');
        $isDryRun = $this->option('dry-run');
        $cutoffDate = now()->subDays($days);

        $this->info("🧹 Cleaning up ads data older than {$days} days...");
        if ($isDryRun) {
            $this->warn('🔍 DRY RUN MODE - No data will be deleted');
        }

        // Clean up raw ads
        $rawAdsCount = DB::table('ads_raw')->where('created_at', '<', $cutoffDate)->count();
        if ($rawAdsCount > 0) {
            if ($isDryRun) {
                $this->info("📥 Would delete {$rawAdsCount} raw ads");
            } else {
                DB::table('ads_raw')->where('created_at', '<', $cutoffDate)->delete();
                $this->info("✅ Deleted {$rawAdsCount} raw ads");
            }
        }

        // Clean up old opportunities (keep high score ones)
        $oldOpportunitiesCount = DB::table('opportunities')
            ->where('created_at', '<', $cutoffDate)
            ->where('score', '<', 0.6)
            ->count();

        if ($oldOpportunitiesCount > 0) {
            if ($isDryRun) {
                $this->info("🎯 Would delete {$oldOpportunitiesCount} low-score opportunities");
            } else {
                DB::table('opportunities')
                    ->where('created_at', '<', $cutoffDate)
                    ->where('score', '<', 0.6)
                    ->delete();
                $this->info("✅ Deleted {$oldOpportunitiesCount} low-score opportunities");
            }
        }

        // Archive old sent opportunities
        $sentOpportunitiesCount = DB::table('opportunities')
            ->where('status', 'sent')
            ->where('pushed_at', '<', $cutoffDate)
            ->count();

        if ($sentOpportunitiesCount > 0) {
            if ($isDryRun) {
                $this->info("📤 Would archive {$sentOpportunitiesCount} sent opportunities");
            } else {
                DB::table('opportunities')
                    ->where('status', 'sent')
                    ->where('pushed_at', '<', $cutoffDate)
                    ->update(['status' => 'archived']);
                $this->info("✅ Archived {$sentOpportunitiesCount} sent opportunities");
            }
        }

        // Clean up old featured ads
        $oldFeaturedAdsCount = DB::table('featured_ads')
            ->where('created_at', '<', $cutoffDate)
            ->where('status', 'sent')
            ->count();

        if ($oldFeaturedAdsCount > 0) {
            if ($isDryRun) {
                $this->info("⭐ Would delete {$oldFeaturedAdsCount} old featured ads");
            } else {
                DB::table('featured_ads')
                    ->where('created_at', '<', $cutoffDate)
                    ->where('status', 'sent')
                    ->delete();
                $this->info("✅ Deleted {$oldFeaturedAdsCount} old featured ads");
            }
        }

        // Clean up failed jobs
        $failedJobsCount = DB::table('failed_jobs')->count();
        if ($failedJobsCount > 0) {
            if ($isDryRun) {
                $this->info("❌ Would delete {$failedJobsCount} failed jobs");
            } else {
                DB::table('failed_jobs')->delete();
                $this->info("✅ Deleted {$failedJobsCount} failed jobs");
            }
        }

        if ($isDryRun) {
            $this->info('🔍 Dry run completed. Use without --dry-run to actually delete data.');
        } else {
            $this->info('✅ Cleanup completed successfully!');
        }

        return 0;
    }
}
