<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class CreateSimpleTestData extends Command
{
    protected $signature = 'ads:create-simple-test';
    protected $description = 'Create simple test data';

    public function handle()
    {
        $this->info('📝 Creating simple test data...');

        // Create test cities
        $tehranId = DB::table('cities')->where('name', 'تهران')->value('id');
        if (!$tehranId) {
            $tehranId = DB::table('cities')->insertGetId([
                'name' => 'تهران',
                'created_at' => now(),
                'updated_at' => now(),
            ]);
            $this->info('✅ Created Tehran city');
        }

        $mashhadId = DB::table('cities')->where('name', 'مشهد')->value('id');
        if (!$mashhadId) {
            $mashhadId = DB::table('cities')->insertGetId([
                'name' => 'مشهد',
                'created_at' => now(),
                'updated_at' => now(),
            ]);
            $this->info('✅ Created Mashhad city');
        }

        // Create test raw ads
        $rawAds = [
            [
                'source' => 'bama',
                'payload_json' => json_encode([
                    'code' => 'test_001',
                    'title' => 'پژو 206 تیپ 2',
                    'brand' => 'پژو',
                    'model' => '206',
                    'year' => 1399,
                    'km' => 85000,
                    'price' => 450,
                    'body_status' => 'ok',
                    'link' => 'https://bama.ir/test/001',
                    'city' => 'تهران'
                ]),
                'fetched_at' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'source' => 'bama',
                'payload_json' => json_encode([
                    'code' => 'test_002',
                    'title' => 'سمند LX',
                    'brand' => 'سمند',
                    'model' => 'LX',
                    'year' => 1398,
                    'km' => 120000,
                    'price' => 320,
                    'body_status' => 'minor',
                    'link' => 'https://bama.ir/test/002',
                    'city' => 'تهران'
                ]),
                'fetched_at' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'source' => 'divar',
                'payload_json' => json_encode([
                    'external_code' => 'divar_001',
                    'title' => 'تیبا 2',
                    'brand' => 'تیبا',
                    'model' => '2',
                    'year' => 1400,
                    'km' => 45000,
                    'price' => 280,
                    'body_status' => 'ok',
                    'link' => 'https://divar.ir/v/divar_001',
                    'city' => 'تهران'
                ]),
                'fetched_at' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ];

        foreach ($rawAds as $ad) {
            try {
                DB::table('ads_raw')->insert($ad);
            } catch (\Throwable $e) {
                if (str_contains($e->getMessage(), 'Duplicate')) {
                    $this->warn('⚠️  Duplicate raw ad skipped: ' . $ad['payload_json']);
                } else {
                    throw $e;
                }
            }
        }

        $this->info('✅ Created ' . count($rawAds) . ' test raw ads');

        // Create test opportunities
        $opportunities = [
            [
                'source' => 'bama',
                'code' => 'test_001',
                'brand' => 'پژو',
                'model' => '206',
                'year' => 1399,
                'km' => 85000,
                'city_id' => $tehranId,
                'price' => 450,
                'body_status' => 'ok',
                'link' => 'https://bama.ir/test/001',
                'dedup_key' => 'bama_test_001',
                'score' => 0.85,
                'reasons_json' => json_encode(['قیمت ۱۵٪ زیر بازار', 'کم‌کار']),
                'status' => 'reviewed',
                'ts_posted' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'source' => 'bama',
                'code' => 'test_002',
                'brand' => 'سمند',
                'model' => 'LX',
                'year' => 1398,
                'km' => 120000,
                'city_id' => $tehranId,
                'price' => 320,
                'body_status' => 'minor',
                'link' => 'https://bama.ir/test/002',
                'dedup_key' => 'bama_test_002',
                'score' => 0.75,
                'reasons_json' => json_encode(['قیمت ۱۰٪ زیر بازار', 'کارکرد مناسب']),
                'status' => 'reviewed',
                'ts_posted' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'source' => 'divar',
                'code' => 'divar_001',
                'brand' => 'تیبا',
                'model' => '2',
                'year' => 1400,
                'km' => 45000,
                'city_id' => $tehranId,
                'price' => 280,
                'body_status' => 'ok',
                'link' => 'https://divar.ir/v/divar_001',
                'dedup_key' => 'divar_test_001',
                'score' => 0.90,
                'reasons_json' => json_encode(['قیمت ۱۵٪ زیر بازار', 'خودرو تازه']),
                'status' => 'reviewed',
                'ts_posted' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ];

        foreach ($opportunities as $opportunity) {
            try {
                DB::table('opportunities')->insert($opportunity);
            } catch (\Throwable $e) {
                if (str_contains($e->getMessage(), 'Duplicate')) {
                    $this->warn('⚠️  Duplicate opportunity skipped: ' . $opportunity['code']);
                } else {
                    throw $e;
                }
            }
        }

        $this->info('✅ Created ' . count($opportunities) . ' test opportunities');

        // Create test featured ad
        DB::table('featured_ads')->insert([
            'title' => 'آگهی ویژه تست',
            'description' => 'این یک آگهی ویژه تست است',
            'price' => 500,
            'city' => 'تهران',
            'link' => 'https://example.com/featured',
            'target_roles' => json_encode(['dealer', 'showroom']),
            'target_cities' => json_encode(['تهران']),
            'scheduled_at' => now()->addHour(),
            'status' => 'draft',
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        $this->info('✅ Created test featured ad');

        // Create test scrape profile
        DB::table('scrape_profiles')->insert([
            'name' => 'تست دیوار تهران',
            'url' => 'https://divar.ir/s/tehran/car',
            'class_name' => 'kt-post-card__body',
            'city' => 'تهران',
            'keywords' => 'خودرو,ماشین',
            'enabled' => true,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        $this->info('✅ Created test scrape profile');

        $this->info('🎉 Simple test data created successfully!');
        return 0;
    }
}
