<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class ShowCarMenu extends Command
{
    protected $signature = 'ads:car-menu';
    protected $description = 'Show car selection menu for price analysis';

    private array $availableCars = [
        '206' => 'پژو 206',
        'ساینا' => 'ساینا',
        'کوییک' => 'کوییک',
        'تیبا' => 'تیبا',
        'تیبا 2' => 'تیبا 2',
        '207' => 'پژو 207',
        'پراید' => 'پراید',
    ];

    public function handle()
    {
        $this->info('🚗 آگهی‌های به قیمت');
        $this->newLine();

        // Main menu
        $mainChoice = $this->choice(
            'چه کاری می‌خواهید انجام دهید؟',
            [
                'view' => 'مشاهده آگهی‌های فعلی',
                'search' => 'پیدا کردن آگهی جدید',
                'exit' => 'خروج'
            ],
            'view'
        );

        if ($mainChoice === 'exit') {
            $this->info('👋 خداحافظ!');
            return 0;
        }

        if ($mainChoice === 'view') {
            return $this->showCurrentAds();
        }

        if ($mainChoice === 'search') {
            return $this->showCarSelectionMenu();
        }

        return 0;
    }

    private function showCurrentAds()
    {
        $this->info('📋 آگهی‌های فعلی:');
        $this->newLine();

        $opportunities = DB::table('opportunities')
            ->orderBy('score', 'desc')
            ->limit(20)
            ->get();

        if ($opportunities->isEmpty()) {
            $this->warn('⚠️  هیچ آگهی‌ای موجود نیست');
            $this->newLine();
            $this->info('💡 برای اضافه کردن آگهی جدید، گزینه "پیدا کردن آگهی جدید" را انتخاب کنید');
            return 0;
        }

        $this->info("📊 تعداد آگهی‌ها: {$opportunities->count()}");
        $this->newLine();

        foreach ($opportunities as $index => $opp) {
            $this->line("  " . ($index + 1) . ". {$opp->brand} {$opp->model} ({$opp->year})");
            $this->line("     💰 قیمت: " . number_format($opp->price) . " میلیون تومان");
            $this->line("     ⭐ امتیاز: " . number_format($opp->score, 2));
            $this->line("     🔗 لینک: {$opp->link}");
            $this->newLine();
        }

        return 0;
    }

    private function showCarSelectionMenu()
    {
        $this->info('🚗 انتخاب خودرو برای جستجو:');
        $this->newLine();

        $selectedCar = $this->choice(
            'لطفاً خودرو مورد نظر را انتخاب کنید:',
            $this->availableCars,
            '206'
        );

        $carName = $this->availableCars[$selectedCar];
        $this->newLine();
        $this->info("✅ شما {$carName} را انتخاب کردید");
        $this->newLine();

        // Ask for confirmation
        if ($this->confirm('آیا می‌خواهید اسکرپ و تحلیل آگهی‌های این خودرو را شروع کنیم؟')) {
            $this->info("🚀 شروع اسکرپ و تحلیل {$carName}...");
            $this->newLine();

            // Run scraping command
            $this->call('ads:scrape-cars', [
                'brand' => $selectedCar,
                '--force' => true
            ]);

            // Show results
            $this->showResults($selectedCar);
        } else {
            $this->info('❌ عملیات لغو شد');
        }

        return 0;
    }

    private function showResults(string $carBrand): void
    {
        $this->newLine();
        $this->info('📊 نتایج تحلیل:');
        $this->newLine();

        // Get opportunities for this car
        $opportunities = DB::table('opportunities')
            ->where('brand', 'like', '%' . $carBrand . '%')
            ->orWhere('brand', 'like', '%' . $this->getBrandMapping($carBrand) . '%')
            ->orderBy('score', 'desc')
            ->get();

        if ($opportunities->isEmpty()) {
            $this->warn('⚠️  هیچ آگهی‌ای برای این خودرو یافت نشد');
            return;
        }

        // Group by score ranges
        $highScore = $opportunities->where('score', '>=', 0.8)->count();
        $mediumScore = $opportunities->whereBetween('score', [0.6, 0.8])->count();
        $lowScore = $opportunities->where('score', '<', 0.6)->count();

        $this->info("📈 توزیع امتیازات:");
        $this->line("  🟢 امتیاز بالا (≥0.8): {$highScore} آگهی");
        $this->line("  🟡 امتیاز متوسط (0.6-0.8): {$mediumScore} آگهی");
        $this->line("  🔴 امتیاز پایین (<0.6): {$lowScore} آگهی");
        $this->newLine();

        // Show price analysis
        $prices = $opportunities->pluck('price')->filter()->sort();
        if ($prices->isNotEmpty()) {
            $minPrice = $prices->first();
            $maxPrice = $prices->last();
            $avgPrice = $prices->avg();

            $this->info("💰 تحلیل قیمت:");
            $this->line("  💵 کمترین قیمت: " . number_format($minPrice) . " میلیون تومان");
            $this->line("  💵 بیشترین قیمت: " . number_format($maxPrice) . " میلیون تومان");
            $this->line("  💵 میانگین قیمت: " . number_format($avgPrice) . " میلیون تومان");
            $this->newLine();
        }

        // Show top opportunities
        $topOpportunities = $opportunities->take(5);
        $this->info("🏆 بهترین آگهی‌ها:");
        $this->newLine();

        foreach ($topOpportunities as $index => $opp) {
            $this->line("  " . ($index + 1) . ". {$opp->brand} {$opp->model} ({$opp->year})");
            $this->line("     💰 قیمت: " . number_format($opp->price) . " میلیون تومان");
            $this->line("     ⭐ امتیاز: " . number_format($opp->score, 2));
            $this->line("     🔗 لینک: {$opp->link}");
            $this->newLine();
        }

        $this->info("✅ تحلیل کامل شد!");
    }

    private function getBrandMapping(string $carBrand): string
    {
        $mappings = [
            '206' => 'peugeot',
            'ساینا' => 'saina',
            'کوییک' => 'quick',
            'تیبا' => 'tiba',
            'تیبا 2' => 'tiba',
            '207' => 'peugeot',
            'پراید' => 'pride',
        ];

        return $mappings[$carBrand] ?? $carBrand;
    }
}
