<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use App\Jobs\BamaPullLatestJob;
use App\Jobs\ScoreOpportunitiesJob;

class TestAdsSystemSimple extends Command
{
    protected $signature = 'ads:test-simple';
    protected $description = 'Simple test of ads system';

    public function handle()
    {
        $this->info('🧪 Testing ads system...');

        // Step 1: Create test data
        $this->info('📝 Creating test data...');
        $this->createTestData();

        // Step 2: Test scoring
        $this->info('🎯 Testing scoring...');
        ScoreOpportunitiesJob::dispatch();

        // Wait a moment
        sleep(2);

        // Step 3: Show results
        $this->showResults();

        $this->info('✅ Test completed!');
        return 0;
    }

    private function createTestData()
    {
        // Create test cities if not exist
        $tehranId = DB::table('cities')->where('name', 'تهران')->value('id');
        if (!$tehranId) {
            $tehranId = DB::table('cities')->insertGetId([
                'name' => 'تهران',
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }

        // Create test raw ads
        $rawAds = [
            [
                'source' => 'bama',
                'payload_json' => json_encode([
                    'code' => 'test_001',
                    'title' => 'پژو 206 تیپ 2',
                    'brand' => 'پژو',
                    'model' => '206',
                    'year' => 1399,
                    'km' => 85000,
                    'price' => 450,
                    'body_status' => 'ok',
                    'link' => 'https://bama.ir/test/001',
                    'city' => 'تهران'
                ]),
                'fetched_at' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'source' => 'bama',
                'payload_json' => json_encode([
                    'code' => 'test_002',
                    'title' => 'سمند LX',
                    'brand' => 'سمند',
                    'model' => 'LX',
                    'year' => 1398,
                    'km' => 120000,
                    'price' => 320,
                    'body_status' => 'minor',
                    'link' => 'https://bama.ir/test/002',
                    'city' => 'تهران'
                ]),
                'fetched_at' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ];

        foreach ($rawAds as $ad) {
            DB::table('ads_raw')->insert($ad);
        }

        $this->info('✅ Created ' . count($rawAds) . ' test raw ads');
    }

    private function showResults()
    {
        $rawCount = DB::table('ads_raw')->count();
        $opportunitiesCount = DB::table('opportunities')->count();
        $reviewedCount = DB::table('opportunities')->where('status', 'reviewed')->count();

        $this->table(
            ['Metric', 'Count'],
            [
                ['Raw Ads', $rawCount],
                ['Opportunities', $opportunitiesCount],
                ['Reviewed', $reviewedCount],
            ]
        );
    }
}
