<?php

namespace App\Filament\Pages;

use App\Models\Setting;
use App\Services\Messaging\TelegramBot;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Illuminate\Support\Facades\Log;

class Settings extends Page implements HasForms
{
    use InteractsWithForms;

    protected static ?string $navigationIcon = 'heroicon-o-cog-6-tooth';
    protected static ?string $navigationLabel = 'تنظیمات';
    protected static ?string $navigationGroup = 'تنظیمات';
    protected static string $view = 'filament.pages.settings';

    public bool $require_phone_verification = false;
    public bool $require_channel_membership = false;
    public string $phone_gate_policy = 'hard_gate';
    public array $channels = [];
    public array $newChannel = [
        'title' => '',
        'chat_id' => '',
        'link' => '',
    ];
    public ?array $testResult = null;

    public function mount(): void
    {
        $this->require_phone_verification = Setting::get('require_phone_verification', false);
        $this->require_channel_membership = Setting::get('require_channel_membership', false);
        $this->phone_gate_policy = Setting::get('phone_gate_policy', 'hard_gate');
        $this->channels = Setting::get('channels', []);
    }

    public function updateSetting(string $key, $value): void
    {
        Setting::set($key, $value, 'boolean', "تنظیم {$key}");

        Notification::make()
            ->title('تنظیمات بروزرسانی شد')
            ->success()
            ->send();
    }

    public function addChannel(): void
    {
        $this->validate([
            'newChannel.title' => 'required|string|max:255',
            'newChannel.chat_id' => 'required|string|max:255',
            'newChannel.link' => 'required|url',
        ]);

        $this->channels[] = $this->newChannel;

        Setting::set('channels', $this->channels, 'json', 'لیست کانال‌های تلگرام');

        $this->newChannel = [
            'title' => '',
            'chat_id' => '',
            'link' => '',
        ];

        Notification::make()
            ->title('کانال اضافه شد')
            ->success()
            ->send();
    }

    public function removeChannel(int $index): void
    {
        if (isset($this->channels[$index])) {
            unset($this->channels[$index]);
            $this->channels = array_values($this->channels);

            Setting::set('channels', $this->channels, 'json', 'لیست کانال‌های تلگرام');

            Notification::make()
                ->title('کانال حذف شد')
                ->success()
                ->send();
        }
    }

    public function testChannel(int $index): void
    {
        if (!isset($this->channels[$index])) {
            return;
        }

        $channel = $this->channels[$index];

        try {
            $bot = app(TelegramBot::class);
            // Test basic connectivity - in real implementation, you'd test membership
            $this->testResult = [
                'channel' => $channel['title'],
                'chat_id' => $channel['chat_id'],
                'status' => 'success',
                'message' => 'کانال قابل دسترسی است',
            ];
        } catch (\Exception $e) {
            $this->testResult = [
                'channel' => $channel['title'],
                'status' => 'error',
                'message' => $e->getMessage(),
            ];

            Log::error('Channel test failed', [
                'channel' => $channel,
                'error' => $e->getMessage(),
            ]);
        }
    }

    public function testWebhook(): void
    {
        try {
            // Send a test request to webhook endpoint
            $response = \Illuminate\Support\Facades\Http::timeout(10)
                ->post(config('app.url') . '/api/telegram/webhook', [
                    'update_type' => 'test',
                    'message' => ['text' => '/test'],
                ]);

            $this->testResult = [
                'endpoint' => config('app.url') . '/api/telegram/webhook',
                'status_code' => $response->status(),
                'response' => $response->body(),
                'status' => $response->successful() ? 'success' : 'error',
            ];
        } catch (\Exception $e) {
            $this->testResult = [
                'endpoint' => config('app.url') . '/api/telegram/webhook',
                'status' => 'error',
                'message' => $e->getMessage(),
            ];
        }
    }

    public function testMembership(): void
    {
        try {
            $bot = app(TelegramBot::class);
            $results = [];

            foreach ($this->channels as $channel) {
                try {
                    // In real implementation, test actual membership
                    $results[] = [
                        'channel' => $channel['title'],
                        'status' => 'success',
                        'message' => 'عضویت قابل بررسی است',
                    ];
                } catch (\Exception $e) {
                    $results[] = [
                        'channel' => $channel['title'],
                        'status' => 'error',
                        'message' => $e->getMessage(),
                    ];
                }
            }

            $this->testResult = [
                'membership_test' => $results,
                'status' => 'completed',
            ];
        } catch (\Exception $e) {
            $this->testResult = [
                'status' => 'error',
                'message' => $e->getMessage(),
            ];
        }
    }
}

