<?php

namespace App\Filament\Resources;

use App\Filament\Resources\FeaturedAdsResource\Pages;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;

class FeaturedAdsResource extends Resource
{
    protected static ?string $model = null; // We'll use DB table directly
    protected static ?string $navigationIcon = 'heroicon-o-star';
    protected static ?string $navigationLabel = 'آگهی‌های ویژه';
    protected static ?string $pluralModelLabel = 'آگهی‌های ویژه';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('اطلاعات آگهی')
                    ->schema([
                        Forms\Components\TextInput::make('title')
                            ->label('عنوان')
                            ->required()
                            ->maxLength(255),
                        Forms\Components\Textarea::make('description')
                            ->label('توضیحات')
                            ->rows(3)
                            ->maxLength(1000),
                        Forms\Components\TextInput::make('price')
                            ->label('قیمت (میلیون تومان)')
                            ->numeric()
                            ->minValue(0),
                        Forms\Components\TextInput::make('city')
                            ->label('شهر')
                            ->maxLength(100),
                        Forms\Components\TextInput::make('link')
                            ->label('لینک آگهی')
                            ->url()
                            ->maxLength(500),
                    ])->columns(2),

                Forms\Components\Section::make('مخاطبان هدف')
                    ->schema([
                        Forms\Components\Select::make('target_roles')
                            ->label('نقش‌های هدف')
                            ->multiple()
                            ->options([
                                'dealer' => 'دلال',
                                'showroom' => 'نمایشگاه‌دار',
                                'newbie' => 'تازه‌کار',
                            ])
                            ->preload(),
                        Forms\Components\Select::make('target_cities')
                            ->label('شهرهای هدف')
                            ->multiple()
                            ->options(function () {
                                return DB::table('cities')->pluck('name', 'name')->toArray();
                            })
                            ->searchable()
                            ->preload(),
                    ])->columns(2),

                Forms\Components\Section::make('زمان‌بندی')
                    ->schema([
                        Forms\Components\DateTimePicker::make('scheduled_at')
                            ->label('زمان ارسال')
                            ->required()
                            ->default(now()->addHour()),
                        Forms\Components\Select::make('status')
                            ->label('وضعیت')
                            ->options([
                                'draft' => 'پیش‌نویس',
                                'scheduled' => 'زمان‌بندی شده',
                                'sent' => 'ارسال شده',
                                'cancelled' => 'لغو شده',
                            ])
                            ->default('draft')
                            ->required(),
                    ])->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->query(DB::table('featured_ads'))
            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->label('عنوان')
                    ->searchable()
                    ->sortable()
                    ->limit(50),
                Tables\Columns\TextColumn::make('price')
                    ->label('قیمت')
                    ->formatStateUsing(fn ($state) => $state ? number_format($state) . ' میلیون' : '-')
                    ->sortable(),
                Tables\Columns\TextColumn::make('city')
                    ->label('شهر')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\BadgeColumn::make('status')
                    ->label('وضعیت')
                    ->formatStateUsing(fn ($state) => match($state) {
                        'draft' => 'پیش‌نویس',
                        'scheduled' => 'زمان‌بندی شده',
                        'sent' => 'ارسال شده',
                        'cancelled' => 'لغو شده',
                        default => $state
                    })
                    ->color(fn ($state) => match($state) {
                        'draft' => 'gray',
                        'scheduled' => 'warning',
                        'sent' => 'success',
                        'cancelled' => 'danger',
                        default => 'gray'
                    }),
                Tables\Columns\TextColumn::make('scheduled_at')
                    ->label('زمان ارسال')
                    ->dateTime()
                    ->sortable(),
                Tables\Columns\TextColumn::make('pushed_at')
                    ->label('ارسال شده')
                    ->dateTime()
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('ایجاد شده')
                    ->dateTime()
                    ->sortable(),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('status')
                    ->label('وضعیت')
                    ->options([
                        'draft' => 'پیش‌نویس',
                        'scheduled' => 'زمان‌بندی شده',
                        'sent' => 'ارسال شده',
                        'cancelled' => 'لغو شده',
                    ]),
                Tables\Filters\Filter::make('scheduled_today')
                    ->label('امروز')
                    ->query(fn (Builder $query) => $query->whereDate('scheduled_at', today())),
            ])
            ->actions([
                Tables\Actions\Action::make('edit')
                    ->label('ویرایش')
                    ->icon('heroicon-o-pencil')
                    ->url(fn ($record) => route('filament.admin.resources.featured-ads.edit', $record->id)),
                Tables\Actions\Action::make('push_now')
                    ->label('ارسال فوری')
                    ->icon('heroicon-o-paper-airplane')
                    ->color('success')
                    ->action(function ($record) {
                        // Dispatch push job immediately
                        \App\Jobs\PushFeaturedAdsJob::dispatch($record->id);

                        // Update status
                        DB::table('featured_ads')
                            ->where('id', $record->id)
                            ->update(['status' => 'sent', 'pushed_at' => now()]);
                    })
                    ->visible(fn ($record) => in_array($record->status, ['draft', 'scheduled'])),
                Tables\Actions\Action::make('cancel')
                    ->label('لغو')
                    ->icon('heroicon-o-x-mark')
                    ->color('danger')
                    ->action(function ($record) {
                        DB::table('featured_ads')
                            ->where('id', $record->id)
                            ->update(['status' => 'cancelled']);
                    })
                    ->visible(fn ($record) => $record->status === 'scheduled'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\BulkAction::make('schedule')
                        ->label('زمان‌بندی')
                        ->action(fn ($records) => $records->each(function ($record) {
                            DB::table('featured_ads')
                                ->where('id', $record->id)
                                ->update(['status' => 'scheduled']);
                        })),
                ]),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListFeaturedAds::route('/'),
            'create' => Pages\CreateFeaturedAd::route('/create'),
            'edit' => Pages\EditFeaturedAd::route('/{record}/edit'),
        ];
    }
}

