<?php

namespace App\Filament\Resources;

use App\Filament\Resources\MenuItemsResource\Pages;
use App\Models\MenuItem;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class MenuItemsResource extends Resource
{
    protected static ?string $model = MenuItem::class;
    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $navigationLabel = 'آیتم‌های منو';
    protected static ?string $modelLabel = 'آیتم منو';
    protected static ?string $pluralModelLabel = 'آیتم‌های منو';
    protected static ?string $navigationGroup = 'تنظیمات';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('اطلاعات پایه')
                    ->schema([
                        Forms\Components\TextInput::make('key')
                            ->label('کلید')
                            ->required()
                            ->unique(ignoreRecord: true)
                            ->placeholder('مثال: opportunities, valuation'),
                        Forms\Components\TextInput::make('title')
                            ->label('عنوان')
                            ->required()
                            ->maxLength(255),
                        Forms\Components\TextInput::make('emoji')
                            ->label('ایموجی')
                            ->placeholder('مثال: 🔎, 💰'),
                        Forms\Components\TextInput::make('ord')
                            ->label('ترتیب')
                            ->numeric()
                            ->default(0)
                            ->required(),
                    ])->columns(2),

                Forms\Components\Section::make('تنظیمات دسترسی')
                    ->schema([
                        Forms\Components\Toggle::make('enabled')
                            ->label('فعال')
                            ->default(true),
                        Forms\Components\Select::make('visible_roles')
                            ->label('نقش‌های قابل مشاهده')
                            ->multiple()
                            ->options([
                                'dealer' => 'دلال',
                                'showroom' => 'نمایشگاه‌دار',
                                'newbie' => 'تازه‌کار',
                            ])
                            ->placeholder('اگر خالی باشد برای همه نمایش داده می‌شود'),
                    ])->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('ord')
                    ->label('ترتیب')
                    ->sortable()
                    ->alignCenter(),
                Tables\Columns\TextColumn::make('key')
                    ->label('کلید')
                    ->searchable()
                    ->sortable()
                    ->copyable(),
                Tables\Columns\TextColumn::make('title')
                    ->label('عنوان')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('emoji')
                    ->label('ایموجی')
                    ->alignCenter(),
                Tables\Columns\IconColumn::make('enabled')
                    ->label('فعال')
                    ->boolean()
                    ->trueIcon('heroicon-o-check-circle')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('danger'),
                Tables\Columns\TextColumn::make('visible_roles')
                    ->label('نقش‌ها')
                    ->formatStateUsing(function ($state) {
                        if (!$state) return 'همه';
                        $roles = is_array($state) ? $state : json_decode($state, true);
                        return implode('، ', array_map(function($role) {
                            return match($role) {
                                'dealer' => 'دلال',
                                'showroom' => 'نمایشگاه‌دار',
                                'newbie' => 'تازه‌کار',
                                default => $role,
                            };
                        }, $roles));
                    })
                    ->badge()
                    ->color('info'),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('ایجاد شده')
                    ->dateTime('Y/m/d')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->label('بروزرسانی')
                    ->dateTime('Y/m/d')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\TernaryFilter::make('enabled')
                    ->label('وضعیت')
                    ->placeholder('همه')
                    ->trueLabel('فعال')
                    ->falseLabel('غیرفعال'),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\Action::make('toggle_enabled')
                    ->label('تغییر وضعیت')
                    ->icon('heroicon-o-arrow-path')
                    ->color('warning')
                    ->action(function (MenuItem $record): void {
                        $record->update(['enabled' => !$record->enabled]);
                    }),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\BulkAction::make('enable_selected')
                        ->label('فعال کردن انتخاب شده‌ها')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->action(function ($records): void {
                            $records->each(function ($record) {
                                $record->update(['enabled' => true]);
                            });
                        }),
                    Tables\Actions\BulkAction::make('disable_selected')
                        ->label('غیرفعال کردن انتخاب شده‌ها')
                        ->icon('heroicon-o-x-circle')
                        ->color('danger')
                        ->action(function ($records): void {
                            $records->each(function ($record) {
                                $record->update(['enabled' => false]);
                            });
                        }),
                ]),
            ])
            ->defaultSort('ord', 'asc')
            ->reorderable('ord');
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListMenuItems::route('/'),
            'create' => Pages\CreateMenuItem::route('/create'),
            'view' => Pages\ViewMenuItem::route('/{record}'),
            'edit' => Pages\EditMenuItem::route('/{record}/edit'),
        ];
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::where('enabled', false)->count();
    }
}

