<?php

namespace App\Filament\Resources;

use App\Filament\Resources\OpportunitiesResource\Pages;
use App\Models\Opportunity;
use App\Models\City;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class OpportunitiesResource extends Resource
{
    protected static ?string $model = Opportunity::class;
    protected static ?string $navigationIcon = 'heroicon-o-bolt';
    protected static ?string $navigationLabel = 'فرصت‌ها';
    protected static ?string $modelLabel = 'فرصت';
    protected static ?string $pluralModelLabel = 'فرصت‌ها';
    protected static ?string $navigationGroup = 'محتوا';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('اطلاعات اصلی')
                    ->schema([
                        Forms\Components\Select::make('source')
                            ->label('منبع')
                            ->options([
                                'bama' => 'باما',
                                'divar' => 'دیوار',
                            ])
                            ->required(),
                        Forms\Components\TextInput::make('code')
                            ->label('کد آگهی')
                            ->required(),
                        Forms\Components\TextInput::make('brand')
                            ->label('برند')
                            ->required(),
                        Forms\Components\TextInput::make('model')
                            ->label('مدل'),
                        Forms\Components\TextInput::make('trim')
                            ->label('تیپ'),
                        Forms\Components\TextInput::make('year')
                            ->label('سال')
                            ->numeric(),
                    ])->columns(2),

                Forms\Components\Section::make('مشخصات فنی')
                    ->schema([
                        Forms\Components\TextInput::make('km')
                            ->label('کارکرد')
                            ->numeric(),
                        Forms\Components\Select::make('city_id')
                            ->label('شهر')
                            ->options(City::pluck('name', 'id'))
                            ->searchable()
                            ->preload(),
                        Forms\Components\TextInput::make('price')
                            ->label('قیمت')
                            ->numeric()
                            ->suffix('میلیون تومان'),
                        Forms\Components\Select::make('body_status')
                            ->label('بدنه')
                            ->options([
                                'excellent' => 'عالی',
                                'good' => 'خوب',
                                'ok' => 'متوسط',
                                'damaged' => 'ضربه‌خورده',
                            ]),
                        Forms\Components\TextInput::make('contact_ref')
                            ->label('شماره تماس'),
                    ])->columns(2),

                Forms\Components\Section::make('ارزیابی و امتیازدهی')
                    ->schema([
                        Forms\Components\TextInput::make('score')
                            ->label('امتیاز')
                            ->numeric()
                            ->step(0.1)
                            ->min(0)
                            ->max(1),
                        Forms\Components\Textarea::make('reasons_json')
                            ->label('دلایل امتیازدهی')
                            ->rows(3)
                            ->placeholder('دلایل امتیازدهی را وارد کنید'),
                        Forms\Components\Select::make('status')
                            ->label('وضعیت')
                            ->options([
                                'draft' => 'پیش‌نویس',
                                'reviewed' => 'بررسی شده',
                                'scheduled' => 'زمان‌بندی شده',
                                'sent' => 'ارسال شده',
                                'archived' => 'بایگانی شده',
                            ])
                            ->default('draft')
                            ->required(),
                    ])->columns(2),

                Forms\Components\Section::make('لینک‌ها و تاریخچه')
                    ->schema([
                        Forms\Components\TextInput::make('link')
                            ->label('لینک آگهی')
                            ->url(),
                        Forms\Components\TextInput::make('dedup_key')
                            ->label('کلید حذف تکرار')
                            ->disabled(),
                        Forms\Components\DateTimePicker::make('ts_posted')
                            ->label('تاریخ انتشار'),
                        Forms\Components\DateTimePicker::make('pushed_at')
                            ->label('تاریخ ارسال'),
                    ])->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('id')
                    ->label('شناسه')
                    ->sortable(),
                Tables\Columns\BadgeColumn::make('source')
                    ->label('منبع')
                    ->formatStateUsing(fn (string $state): string => match($state) {
                        'bama' => 'باما',
                        'divar' => 'دیوار',
                    })
                    ->color(fn (string $state): string => match($state) {
                        'bama' => 'blue',
                        'divar' => 'green',
                    }),
                Tables\Columns\TextColumn::make('brand')
                    ->label('برند')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('model')
                    ->label('مدل')
                    ->searchable(),
                Tables\Columns\TextColumn::make('year')
                    ->label('سال')
                    ->sortable(),
                Tables\Columns\TextColumn::make('city.name')
                    ->label('شهر')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('price')
                    ->label('قیمت')
                    ->formatStateUsing(fn (?int $state): string => $state ? number_format($state) . 'M' : '-')
                    ->sortable(),
                Tables\Columns\TextColumn::make('score')
                    ->label('امتیاز')
                    ->formatStateUsing(fn (?float $state): string => $state ? number_format($state, 2) : '-')
                    ->sortable()
                    ->color(fn (?float $state): string => match(true) {
                        $state >= 0.8 => 'success',
                        $state >= 0.6 => 'warning',
                        default => 'danger',
                    }),
                Tables\Columns\BadgeColumn::make('status')
                    ->label('وضعیت')
                    ->formatStateUsing(fn (string $state): string => match($state) {
                        'draft' => 'پیش‌نویس',
                        'reviewed' => 'بررسی شده',
                        'scheduled' => 'زمان‌بندی شده',
                        'sent' => 'ارسال شده',
                        'archived' => 'بایگانی شده',
                    })
                    ->color(fn (string $state): string => match($state) {
                        'draft' => 'gray',
                        'reviewed' => 'blue',
                        'scheduled' => 'yellow',
                        'sent' => 'green',
                        'archived' => 'gray',
                    }),
                Tables\Columns\TextColumn::make('ts_posted')
                    ->label('تاریخ انتشار')
                    ->dateTime('Y/m/d H:i')
                    ->sortable(),
                Tables\Columns\TextColumn::make('pushed_at')
                    ->label('ارسال شده')
                    ->dateTime('Y/m/d H:i')
                    ->sortable(),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('source')
                    ->label('منبع')
                    ->options([
                        'bama' => 'باما',
                        'divar' => 'دیوار',
                    ]),
                Tables\Filters\SelectFilter::make('city_id')
                    ->label('شهر')
                    ->options(City::pluck('name', 'id')),
                Tables\Filters\SelectFilter::make('brand')
                    ->label('برند')
                    ->options(function () {
                        return Opportunity::distinct()->pluck('brand', 'brand')->filter()->toArray();
                    }),
                Tables\Filters\SelectFilter::make('status')
                    ->label('وضعیت')
                    ->options([
                        'draft' => 'پیش‌نویس',
                        'reviewed' => 'بررسی شده',
                        'scheduled' => 'زمان‌بندی شده',
                        'sent' => 'ارسال شده',
                        'archived' => 'بایگانی شده',
                    ]),
                Tables\Filters\Filter::make('high_score')
                    ->label('امتیاز بالا (≥۰.۸)')
                    ->query(fn (Builder $query): Builder => $query->where('score', '>=', 0.8)),
                Tables\Filters\Filter::make('recent')
                    ->label('آگهی‌های اخیر (۷ روز)')
                    ->query(fn (Builder $query): Builder => $query->where('ts_posted', '>=', now()->subDays(7))),
                Tables\Filters\Filter::make('pushed')
                    ->label('ارسال شده')
                    ->query(fn (Builder $query): Builder => $query->whereNotNull('pushed_at')),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\Action::make('approve')
                    ->label('تأیید')
                    ->icon('heroicon-o-check-circle')
                    ->color('success')
                    ->action(function (Opportunity $record): void {
                        $record->update(['status' => 'reviewed']);
                    })
                    ->visible(fn (Opportunity $record): bool => $record->status === 'draft'),
                Tables\Actions\Action::make('archive')
                    ->label('بایگانی')
                    ->icon('heroicon-o-archive-box')
                    ->color('gray')
                    ->action(function (Opportunity $record): void {
                        $record->update(['status' => 'archived']);
                    })
                    ->visible(fn (Opportunity $record): bool => in_array($record->status, ['draft', 'reviewed'])),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\BulkAction::make('approve_selected')
                        ->label('تأیید انتخاب شده‌ها')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->action(function ($records): void {
                            $records->each(function ($record) {
                                $record->update(['status' => 'reviewed']);
                            });
                        }),
                    Tables\Actions\BulkAction::make('archive_selected')
                        ->label('بایگانی انتخاب شده‌ها')
                        ->icon('heroicon-o-archive-box')
                        ->color('gray')
                        ->action(function ($records): void {
                            $records->each(function ($record) {
                                $record->update(['status' => 'archived']);
                            });
                        }),
                ]),
            ])
            ->defaultSort('ts_posted', 'desc');
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListOpportunities::route('/'),
            'create' => Pages\CreateOpportunity::route('/create'),
            'view' => Pages\ViewOpportunity::route('/{record}'),
            'edit' => Pages\EditOpportunity::route('/{record}/edit'),
        ];
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::where('status', 'draft')->count();
    }
}

