<?php

namespace App\Filament\Resources;

use App\Filament\Resources\OpportunityResource\Pages;
use App\Filament\Resources\OpportunityResource\RelationManagers;
use App\Models\Opportunity;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Support\Colors\Color;

class OpportunityResource extends Resource
{
    protected static ?string $model = Opportunity::class;

    protected static ?string $navigationIcon = 'heroicon-o-currency-dollar';

    protected static ?string $navigationLabel = 'فرصت‌ها';

    protected static ?string $pluralModelLabel = 'فرصت‌ها';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('اطلاعات اصلی')
                    ->schema([
                        Forms\Components\TextInput::make('brand')
                            ->label('برند')
                            ->required(),
                        Forms\Components\TextInput::make('model')
                            ->label('مدل')
                            ->required(),
                        Forms\Components\TextInput::make('year')
                            ->label('سال')
                            ->numeric()
                            ->required(),
                        Forms\Components\TextInput::make('km')
                            ->label('کارکرد')
                            ->numeric(),
                        Forms\Components\TextInput::make('price')
                            ->label('قیمت (میلیون)')
                            ->numeric()
                            ->required(),
                    ])->columns(3),

                Forms\Components\Section::make('وضعیت و امتیاز')
                    ->schema([
                        Forms\Components\Select::make('status')
                            ->label('وضعیت')
                            ->options([
                                'draft' => 'پیش‌نویس',
                                'reviewed' => 'بررسی شده',
                                'scheduled' => 'زمان‌بندی شده',
                                'sent' => 'ارسال شده',
                                'archived' => 'آرشیو شده',
                            ])
                            ->required(),
                        Forms\Components\TextInput::make('score')
                            ->label('امتیاز')
                            ->numeric()
                            ->step(0.1)
                            ->minValue(0)
                            ->maxValue(1),
                        Forms\Components\Select::make('body_status')
                            ->label('وضعیت بدنه')
                            ->options([
                                'ok' => 'سالم',
                                'minor' => 'جزئی',
                                'major' => 'نیاز به تعمیر',
                                'accident' => 'تصادفی',
                            ]),
                    ])->columns(3),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('brand')
                    ->label('برند')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('model')
                    ->label('مدل')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('year')
                    ->label('سال')
                    ->sortable(),
                Tables\Columns\TextColumn::make('km')
                    ->label('کارکرد')
                    ->formatStateUsing(fn ($state) => number_format($state) . ' کیلومتر')
                    ->sortable(),
                Tables\Columns\TextColumn::make('price')
                    ->label('قیمت')
                    ->formatStateUsing(fn ($state) => number_format($state) . ' میلیون')
                    ->sortable(),
                Tables\Columns\TextColumn::make('score')
                    ->label('امتیاز')
                    ->formatStateUsing(fn ($state) => number_format($state * 100) . '%')
                    ->color(fn ($state) => $state >= 0.8 ? 'success' : ($state >= 0.6 ? 'warning' : 'danger'))
                    ->sortable(),
                Tables\Columns\BadgeColumn::make('status')
                    ->label('وضعیت')
                    ->formatStateUsing(fn ($state) => match($state) {
                        'draft' => 'پیش‌نویس',
                        'reviewed' => 'بررسی شده',
                        'scheduled' => 'زمان‌بندی شده',
                        'sent' => 'ارسال شده',
                        'archived' => 'آرشیو شده',
                        default => $state
                    })
                    ->color(fn ($state) => match($state) {
                        'draft' => 'gray',
                        'reviewed' => 'info',
                        'scheduled' => 'warning',
                        'sent' => 'success',
                        'archived' => 'danger',
                        default => 'gray'
                    }),
                Tables\Columns\TextColumn::make('city.name')
                    ->label('شهر')
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('ایجاد شده')
                    ->dateTime()
                    ->sortable(),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('status')
                    ->label('وضعیت')
                    ->options([
                        'draft' => 'پیش‌نویس',
                        'reviewed' => 'بررسی شده',
                        'scheduled' => 'زمان‌بندی شده',
                        'sent' => 'ارسال شده',
                        'archived' => 'آرشیو شده',
                    ]),
                Tables\Filters\SelectFilter::make('brand')
                    ->label('برند')
                    ->options(fn () => \App\Models\Opportunity::distinct()->pluck('brand', 'brand')->toArray()),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\Action::make('push')
                    ->label('ارسال')
                    ->icon('heroicon-o-paper-airplane')
                    ->color('success')
                    ->action(function ($record) {
                        $record->update(['status' => 'sent', 'pushed_at' => now()]);
                    })
                    ->visible(fn ($record) => in_array($record->status, ['draft', 'reviewed', 'scheduled'])),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\BulkAction::make('mark_reviewed')
                        ->label('علامت‌گذاری به عنوان بررسی شده')
                        ->action(fn ($records) => $records->each->update(['status' => 'reviewed']))
                        ->deselectRecordsAfterCompletion(),
                ]),
            ])
            ->defaultSort('score', 'desc');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListOpportunities::route('/'),
            'create' => Pages\CreateOpportunity::route('/create'),
            'edit' => Pages\EditOpportunity::route('/{record}/edit'),
        ];
    }
}
