<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PromptConfigsResource\Pages;
use App\Models\PromptConfig;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

class PromptConfigsResource extends Resource
{
    protected static ?string $model = PromptConfig::class;
    protected static ?string $navigationIcon = 'heroicon-o-chat-bubble-left-right';
    protected static ?string $navigationLabel = 'تنظیمات پرامپت‌ها';
    protected static ?string $modelLabel = 'تنظیم پرامپت';
    protected static ?string $pluralModelLabel = 'تنظیمات پرامپت‌ها';
    protected static ?string $navigationGroup = 'تنظیمات';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('تنظیمات پایه')
                    ->schema([
                        Forms\Components\Select::make('module')
                            ->label('ماژول')
                            ->options([
                                'valuation' => 'قیمت‌یار',
                                'negotiation' => 'تمرین مذاکره',
                                'expert' => 'سؤالات کارشناسی',
                                'legal' => 'سؤالات حقوقی',
                                'inspection' => 'بازدید',
                                'instagram' => 'ایده اینستاگرام',
                            ])
                            ->required()
                            ->unique(ignoreRecord: true),
                        Forms\Components\Select::make('model')
                            ->label('مدل OpenAI')
                            ->options([
                                'gpt-4o-mini' => 'GPT-4o Mini',
                                'gpt-4o' => 'GPT-4o',
                                'gpt-3.5-turbo' => 'GPT-3.5 Turbo',
                            ])
                            ->default('gpt-4o-mini')
                            ->required(),
                        Forms\Components\Toggle::make('enable_file_search')
                            ->label('فعال‌سازی جستجوی فایل')
                            ->default(true),
                    ])->columns(3),

                Forms\Components\Section::make('پارامترهای مدل')
                    ->schema([
                        Forms\Components\TextInput::make('temperature')
                            ->label('دما (Temperature)')
                            ->numeric()
                            ->step(0.1)
                            ->min(0)
                            ->max(2)
                            ->default(0.7)
                            ->required(),
                        Forms\Components\TextInput::make('top_p')
                            ->label('Top P')
                            ->numeric()
                            ->step(0.1)
                            ->min(0)
                            ->max(1)
                            ->default(1.0)
                            ->required(),
                    ])->columns(2),

                Forms\Components\Section::make('پرامپت‌ها')
                    ->schema([
                        Forms\Components\Textarea::make('system_prompt')
                            ->label('پرامپت سیستمی')
                            ->rows(6)
                            ->placeholder('پرامپت سیستمی برای این ماژول...')
                            ->columnSpanFull(),
                        Forms\Components\Textarea::make('user_prompt_template')
                            ->label('قالب پرامپت کاربر')
                            ->rows(4)
                            ->placeholder('قالب پرامپت کاربر با متغیرها...')
                            ->columnSpanFull(),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('module')
                    ->label('ماژول')
                    ->formatStateUsing(fn (string $state): string => match($state) {
                        'valuation' => '💰 قیمت‌یار',
                        'negotiation' => '🗣️ تمرین مذاکره',
                        'expert' => '🔧 سؤالات کارشناسی',
                        'legal' => '⚖️ سؤالات حقوقی',
                        'inspection' => '🔍 بازدید',
                        'instagram' => '📷 ایده اینستاگرام',
                        default => $state,
                    })
                    ->sortable(),
                Tables\Columns\TextColumn::make('model')
                    ->label('مدل')
                    ->formatStateUsing(fn (string $state): string => match($state) {
                        'gpt-4o-mini' => 'GPT-4o Mini',
                        'gpt-4o' => 'GPT-4o',
                        'gpt-3.5-turbo' => 'GPT-3.5 Turbo',
                        default => $state,
                    })
                    ->badge()
                    ->color(fn (string $state): string => match($state) {
                        'gpt-4o' => 'success',
                        'gpt-4o-mini' => 'warning',
                        'gpt-3.5-turbo' => 'gray',
                    }),
                Tables\Columns\IconColumn::make('enable_file_search')
                    ->label('جستجوی فایل')
                    ->boolean()
                    ->trueIcon('heroicon-o-document-magnifying-glass')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('gray'),
                Tables\Columns\TextColumn::make('temperature')
                    ->label('دما')
                    ->formatStateUsing(fn (float $state): string => number_format($state, 1))
                    ->sortable(),
                Tables\Columns\TextColumn::make('top_p')
                    ->label('Top P')
                    ->formatStateUsing(fn (float $state): string => number_format($state, 1))
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('ایجاد شده')
                    ->dateTime('Y/m/d')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->label('بروزرسانی')
                    ->dateTime('Y/m/d')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('module')
                    ->label('ماژول')
                    ->options([
                        'valuation' => 'قیمت‌یار',
                        'negotiation' => 'تمرین مذاکره',
                        'expert' => 'سؤالات کارشناسی',
                        'legal' => 'سؤالات حقوقی',
                        'inspection' => 'بازدید',
                        'instagram' => 'ایده اینستاگرام',
                    ]),
                Tables\Filters\SelectFilter::make('model')
                    ->label('مدل')
                    ->options([
                        'gpt-4o-mini' => 'GPT-4o Mini',
                        'gpt-4o' => 'GPT-4o',
                        'gpt-3.5-turbo' => 'GPT-3.5 Turbo',
                    ]),
                Tables\Filters\TernaryFilter::make('enable_file_search')
                    ->label('جستجوی فایل')
                    ->placeholder('همه')
                    ->trueLabel('فعال')
                    ->falseLabel('غیرفعال'),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\Action::make('test_prompt')
                    ->label('تست پرامپت')
                    ->icon('heroicon-o-play')
                    ->color('info')
                    ->form([
                        Forms\Components\Textarea::make('test_input')
                            ->label('ورودی تست')
                            ->required()
                            ->rows(3)
                            ->placeholder('ورودی آزمایشی برای تست پرامپت...'),
                    ])
                    ->action(function (PromptConfig $record, array $data) {
                        // In real implementation, you'd call the AI service
                        // For now, just show a placeholder response
                        \Filament\Notifications\Notification::make()
                            ->title('تست پرامپت')
                            ->body('پاسخ آزمایشی: پرامپت به درستی تنظیم شده است.')
                            ->success()
                            ->send();
                    }),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\BulkAction::make('enable_file_search')
                        ->label('فعال‌سازی جستجوی فایل')
                        ->icon('heroicon-o-document-magnifying-glass')
                        ->color('success')
                        ->action(function ($records): void {
                            $records->each(function ($record) {
                                $record->update(['enable_file_search' => true]);
                            });
                        }),
                    Tables\Actions\BulkAction::make('disable_file_search')
                        ->label('غیرفعال‌سازی جستجوی فایل')
                        ->icon('heroicon-o-x-circle')
                        ->color('danger')
                        ->action(function ($records): void {
                            $records->each(function ($record) {
                                $record->update(['enable_file_search' => false]);
                            });
                        }),
                ]),
            ])
            ->defaultSort('module', 'asc');
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPromptConfigs::route('/'),
            'create' => Pages\CreatePromptConfig::route('/create'),
            'view' => Pages\ViewPromptConfig::route('/{record}'),
            'edit' => Pages\EditPromptConfig::route('/{record}/edit'),
        ];
    }
}

