<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ScrapeProfilesResource\Pages;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;

class ScrapeProfilesResource extends Resource
{
    protected static ?string $model = null; // We'll use DB table directly
    protected static ?string $navigationIcon = 'heroicon-o-magnifying-glass';
    protected static ?string $navigationLabel = 'پروفایل‌های Scraping';
    protected static ?string $pluralModelLabel = 'پروفایل‌های Scraping';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('اطلاعات پروفایل')
                    ->schema([
                        Forms\Components\TextInput::make('name')
                            ->label('نام پروفایل')
                            ->required()
                            ->maxLength(255),
                        Forms\Components\TextInput::make('url')
                            ->label('URL هدف')
                            ->url()
                            ->required()
                            ->maxLength(500),
                        Forms\Components\TextInput::make('class_name')
                            ->label('CSS Class')
                            ->maxLength(255)
                            ->helperText('کلاس CSS برای استخراج آیتم‌ها'),
                        Forms\Components\TextInput::make('element_id')
                            ->label('Element ID')
                            ->maxLength(255)
                            ->helperText('ID المان خاص برای استخراج'),
                    ])->columns(2),

                Forms\Components\Section::make('تنظیمات فیلتر')
                    ->schema([
                        Forms\Components\TextInput::make('city')
                            ->label('شهر')
                            ->maxLength(100)
                            ->helperText('شهر هدف برای فیلتر کردن نتایج'),
                        Forms\Components\TextInput::make('keywords')
                            ->label('کلمات کلیدی')
                            ->maxLength(500)
                            ->helperText('کلمات کلیدی جدا شده با کاما'),
                        Forms\Components\Toggle::make('enabled')
                            ->label('فعال')
                            ->default(true),
                    ])->columns(3),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->query(DB::table('scrape_profiles'))
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->label('نام')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('url')
                    ->label('URL')
                    ->limit(50)
                    ->searchable(),
                Tables\Columns\TextColumn::make('class_name')
                    ->label('CSS Class')
                    ->limit(30),
                Tables\Columns\TextColumn::make('city')
                    ->label('شهر')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('keywords')
                    ->label('کلمات کلیدی')
                    ->limit(30),
                Tables\Columns\IconColumn::make('enabled')
                    ->label('فعال')
                    ->boolean(),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('ایجاد شده')
                    ->dateTime()
                    ->sortable(),
            ])
            ->filters([
                Tables\Filters\TernaryFilter::make('enabled')
                    ->label('فعال')
                    ->boolean()
                    ->trueLabel('فعال')
                    ->falseLabel('غیرفعال')
                    ->native(false),
                Tables\Filters\Filter::make('has_city')
                    ->label('دارای شهر')
                    ->query(fn (Builder $query) => $query->whereNotNull('city')),
            ])
            ->actions([
                Tables\Actions\Action::make('edit')
                    ->label('ویرایش')
                    ->icon('heroicon-o-pencil')
                    ->url(fn ($record) => route('filament.admin.resources.scrape-profiles.edit', $record->id)),
                Tables\Actions\Action::make('test')
                    ->label('تست')
                    ->icon('heroicon-o-play')
                    ->color('info')
                    ->action(function ($record) {
                        // Test the scraper profile
                        try {
                            $client = app(\App\Services\Ads\MajidApiClient::class);
                            $data = $client->divarScrape($record->url);

                            if (!empty($data)) {
                                \Filament\Notifications\Notification::make()
                                    ->title('تست موفق')
                                    ->body('پروفایل با موفقیت تست شد. ' . count($data) . ' آیتم یافت شد.')
                                    ->success()
                                    ->send();
                            } else {
                                \Filament\Notifications\Notification::make()
                                    ->title('تست ناموفق')
                                    ->body('هیچ داده‌ای یافت نشد.')
                                    ->warning()
                                    ->send();
                            }
                        } catch (\Throwable $e) {
                            \Filament\Notifications\Notification::make()
                                ->title('خطا در تست')
                                ->body('خطا: ' . $e->getMessage())
                                ->danger()
                                ->send();
                        }
                    }),
                Tables\Actions\Action::make('toggle_enabled')
                    ->label(fn ($record) => $record->enabled ? 'غیرفعال' : 'فعال')
                    ->icon(fn ($record) => $record->enabled ? 'heroicon-o-x-mark' : 'heroicon-o-check')
                    ->color(fn ($record) => $record->enabled ? 'danger' : 'success')
                    ->action(function ($record) {
                        DB::table('scrape_profiles')
                            ->where('id', $record->id)
                            ->update(['enabled' => !$record->enabled]);
                    }),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\BulkAction::make('enable')
                        ->label('فعال کردن')
                        ->action(fn ($records) => $records->each(function ($record) {
                            DB::table('scrape_profiles')
                                ->where('id', $record->id)
                                ->update(['enabled' => true]);
                        })),
                    Tables\Actions\BulkAction::make('disable')
                        ->label('غیرفعال کردن')
                        ->action(fn ($records) => $records->each(function ($record) {
                            DB::table('scrape_profiles')
                                ->where('id', $record->id)
                                ->update(['enabled' => false]);
                        })),
                ]),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListScrapeProfiles::route('/'),
            'create' => Pages\CreateScrapeProfile::route('/create'),
            'edit' => Pages\EditScrapeProfile::route('/{record}/edit'),
        ];
    }
}

