<?php

namespace App\Filament\Resources;

use App\Filament\Resources\TicketsResource\Pages;
use App\Models\Ticket;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class TicketsResource extends Resource
{
    protected static ?string $model = Ticket::class;
    protected static ?string $navigationIcon = 'heroicon-o-inbox';
    protected static ?string $navigationLabel = 'تیکت‌های پشتیبانی';
    protected static ?string $modelLabel = 'تیکت';
    protected static ?string $pluralModelLabel = 'تیکت‌ها';
    protected static ?string $navigationGroup = 'پشتیبانی';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('اطلاعات تیکت')
                    ->schema([
                        Forms\Components\Select::make('user_id')
                            ->relationship('user', 'tg_id')
                            ->label('کاربر')
                            ->searchable()
                            ->preload()
                            ->required(),
                        Forms\Components\TextInput::make('subject')
                            ->label('موضوع')
                            ->required()
                            ->maxLength(255),
                        Forms\Components\Select::make('priority')
                            ->label('اولویت')
                            ->options([
                                'low' => 'پایین',
                                'medium' => 'متوسط',
                                'high' => 'بالا',
                            ])
                            ->default('medium')
                            ->required(),
                        Forms\Components\Select::make('status')
                            ->label('وضعیت')
                            ->options([
                                'open' => 'باز',
                                'pending' => 'در انتظار پاسخ',
                                'closed' => 'بسته شده',
                            ])
                            ->default('open')
                            ->required(),
                    ])->columns(2),

                Forms\Components\Section::make('محتوای تیکت')
                    ->schema([
                        Forms\Components\Textarea::make('body')
                            ->label('متن تیکت')
                            ->rows(6)
                            ->required(),
                        Forms\Components\Textarea::make('admin_last_reply_text')
                            ->label('آخرین پاسخ ادمین')
                            ->rows(4)
                            ->placeholder('پاسخ خود را اینجا وارد کنید...'),
                        Forms\Components\DateTimePicker::make('admin_replied_at')
                            ->label('زمان پاسخ'),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('id')
                    ->label('شناسه')
                    ->sortable(),
                Tables\Columns\TextColumn::make('user.tg_id')
                    ->label('کاربر')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('subject')
                    ->label('موضوع')
                    ->searchable()
                    ->sortable()
                    ->limit(40)
                    ->tooltip(function (Tables\Columns\TextColumn $column): ?string {
                        $state = $column->getState();
                        return strlen($state) > 40 ? $state : null;
                    }),
                Tables\Columns\BadgeColumn::make('priority')
                    ->label('اولویت')
                    ->formatStateUsing(fn (string $state): string => match($state) {
                        'low' => 'پایین',
                        'medium' => 'متوسط',
                        'high' => 'بالا',
                    })
                    ->color(fn (string $state): string => match($state) {
                        'high' => 'danger',
                        'medium' => 'warning',
                        'low' => 'success',
                    }),
                Tables\Columns\BadgeColumn::make('status')
                    ->label('وضعیت')
                    ->formatStateUsing(fn (string $state): string => match($state) {
                        'open' => 'باز',
                        'pending' => 'در انتظار',
                        'closed' => 'بسته',
                    })
                    ->color(fn (string $state): string => match($state) {
                        'open' => 'success',
                        'pending' => 'warning',
                        'closed' => 'gray',
                    }),
                Tables\Columns\IconColumn::make('admin_replied_at')
                    ->label('پاسخ داده شده')
                    ->boolean()
                    ->trueIcon('heroicon-o-check-circle')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('danger'),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('ایجاد شده')
                    ->dateTime('Y/m/d H:i')
                    ->sortable(),
                Tables\Columns\TextColumn::make('admin_replied_at')
                    ->label('پاسخ داده شده')
                    ->dateTime('Y/m/d H:i')
                    ->sortable()
                    ->placeholder('پاسخ داده نشده'),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('status')
                    ->label('وضعیت')
                    ->options([
                        'open' => 'باز',
                        'pending' => 'در انتظار پاسخ',
                        'closed' => 'بسته شده',
                    ]),
                Tables\Filters\SelectFilter::make('priority')
                    ->label('اولویت')
                    ->options([
                        'high' => 'بالا',
                        'medium' => 'متوسط',
                        'low' => 'پایین',
                    ]),
                Tables\Filters\Filter::make('unanswered')
                    ->label('بدون پاسخ')
                    ->query(fn (Builder $query): Builder => $query->whereNull('admin_replied_at')),
                Tables\Filters\Filter::make('recent')
                    ->label('تیکت‌های اخیر (۲۴ ساعت)')
                    ->query(fn (Builder $query): Builder => $query->where('created_at', '>=', now()->subDay())),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\Action::make('reply')
                    ->label('پاسخ')
                    ->icon('heroicon-o-chat-bubble-left-right')
                    ->color('info')
                    ->form([
                        Forms\Components\Textarea::make('reply_text')
                            ->label('متن پاسخ')
                            ->required()
                            ->rows(4),
                    ])
                    ->action(function (Ticket $record, array $data): void {
                        $record->update([
                            'admin_last_reply_text' => $data['reply_text'],
                            'admin_replied_at' => now(),
                            'status' => 'pending',
                        ]);
                    })
                    ->visible(fn (Ticket $record): bool => $record->status !== 'closed'),
                Tables\Actions\Action::make('close')
                    ->label('بستن تیکت')
                    ->icon('heroicon-o-x-mark')
                    ->color('success')
                    ->requiresConfirmation()
                    ->action(function (Ticket $record): void {
                        $record->update(['status' => 'closed']);
                    })
                    ->visible(fn (Ticket $record): bool => $record->status !== 'closed'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\BulkAction::make('close_selected')
                        ->label('بستن تیکت‌های انتخاب شده')
                        ->icon('heroicon-o-x-mark')
                        ->color('success')
                        ->action(function ($records): void {
                            $records->each(function ($record) {
                                $record->update(['status' => 'closed']);
                            });
                        }),
                    Tables\Actions\BulkAction::make('mark_pending')
                        ->label('علامت‌گذاری به عنوان در انتظار')
                        ->icon('heroicon-o-clock')
                        ->color('warning')
                        ->action(function ($records): void {
                            $records->each(function ($record) {
                                $record->update(['status' => 'pending']);
                            });
                        }),
                ]),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTickets::route('/'),
            'create' => Pages\CreateTicket::route('/create'),
            'view' => Pages\ViewTicket::route('/{record}'),
            'edit' => Pages\EditTicket::route('/{record}/edit'),
        ];
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::where('status', 'open')->count();
    }
}

