<?php

namespace App\Filament\Resources;

use App\Filament\Resources\VectorStoresResource\Pages;
use App\Models\VectorStore;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

class VectorStoresResource extends Resource
{
    protected static ?string $model = VectorStore::class;
    protected static ?string $navigationIcon = 'heroicon-o-rectangle-group';
    protected static ?string $navigationLabel = 'ویکتور استورها';
    protected static ?string $modelLabel = 'ویکتور استور';
    protected static ?string $pluralModelLabel = 'ویکتور استورها';
    protected static ?string $navigationGroup = 'تنظیمات';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('اطلاعات ویکتور استور')
                    ->schema([
                        Forms\Components\TextInput::make('name')
                            ->label('نام')
                            ->required()
                            ->unique(ignoreRecord: true)
                            ->placeholder('مثال: VS_MAIN, VS_INSTAGRAM'),
                        Forms\Components\TextInput::make('openai_vector_store_id')
                            ->label('شناسه OpenAI')
                            ->placeholder('شناسه ویکتور استور در OpenAI'),
                        Forms\Components\Toggle::make('is_active')
                            ->label('فعال')
                            ->default(true),
                        Forms\Components\Textarea::make('description')
                            ->label('توضیحات')
                            ->rows(3)
                            ->placeholder('توضیحات مربوط به این ویکتور استور...'),
                    ])->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->label('نام')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('openai_vector_store_id')
                    ->label('شناسه OpenAI')
                    ->searchable()
                    ->copyable()
                    ->limit(20)
                    ->tooltip(function (Tables\Columns\TextColumn $column): ?string {
                        $state = $column->getState();
                        return strlen($state) > 20 ? $state : null;
                    }),
                Tables\Columns\IconColumn::make('is_active')
                    ->label('فعال')
                    ->boolean()
                    ->trueIcon('heroicon-o-check-circle')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('danger'),
                Tables\Columns\TextColumn::make('file_count')
                    ->label('تعداد فایل')
                    ->formatStateUsing(fn (int $state): string => number_format($state) . ' فایل')
                    ->sortable(),
                Tables\Columns\TextColumn::make('description')
                    ->label('توضیحات')
                    ->limit(30)
                    ->tooltip(function (Tables\Columns\TextColumn $column): ?string {
                        $state = $column->getState();
                        return strlen($state) > 30 ? $state : null;
                    })
                    ->placeholder('بدون توضیحات'),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('ایجاد شده')
                    ->dateTime('Y/m/d')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->label('بروزرسانی')
                    ->dateTime('Y/m/d')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\TernaryFilter::make('is_active')
                    ->label('وضعیت')
                    ->placeholder('همه')
                    ->trueLabel('فعال')
                    ->falseLabel('غیرفعال'),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\Action::make('toggle_active')
                    ->label('تغییر وضعیت')
                    ->icon('heroicon-o-arrow-path')
                    ->color('warning')
                    ->action(function (VectorStore $record): void {
                        $record->update(['is_active' => !$record->is_active]);
                    }),
                Tables\Actions\Action::make('view_files')
                    ->label('مشاهده فایل‌ها')
                    ->icon('heroicon-o-document')
                    ->color('info')
                    ->url(fn (VectorStore $record): string => route('filament.admin.resources.vector-stores.view', $record)),
                Tables\Actions\Action::make('sync_files')
                    ->label('همگام‌سازی فایل‌ها')
                    ->icon('heroicon-o-arrow-path-circle')
                    ->color('success')
                    ->action(function (VectorStore $record) {
                        // In real implementation, sync with OpenAI
                        \Filament\Notifications\Notification::make()
                            ->title('همگام‌سازی فایل‌ها')
                            ->body('فایل‌ها با موفقیت همگام‌سازی شدند.')
                            ->success()
                            ->send();
                    }),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\BulkAction::make('activate_selected')
                        ->label('فعال کردن انتخاب شده‌ها')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->action(function ($records): void {
                            $records->each(function ($record) {
                                $record->update(['is_active' => true]);
                            });
                        }),
                    Tables\Actions\BulkAction::make('deactivate_selected')
                        ->label('غیرفعال کردن انتخاب شده‌ها')
                        ->icon('heroicon-o-x-circle')
                        ->color('danger')
                        ->action(function ($records): void {
                            $records->each(function ($record) {
                                $record->update(['is_active' => false]);
                            });
                        }),
                ]),
            ])
            ->defaultSort('name', 'asc');
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListVectorStores::route('/'),
            'create' => Pages\CreateVectorStore::route('/create'),
            'view' => Pages\ViewVectorStore::route('/{record}'),
            'edit' => Pages\EditVectorStore::route('/{record}/edit'),
        ];
    }
}

