<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PushSchedulerJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function handle(): void
    {
        try {
            $now = now();
            $processedCount = 0;

            // Process scheduled opportunities
            $this->processScheduledOpportunities($now, $processedCount);

            // Process scheduled featured ads
            $this->processScheduledFeaturedAds($now, $processedCount);

            Log::info('PushSchedulerJob: Completed', [
                'processed_count' => $processedCount,
                'timestamp' => $now->toDateTimeString()
            ]);

        } catch (\Throwable $e) {
            Log::error('PushSchedulerJob failed', [
                'error' => $e->getMessage(),
                'line' => $e->getLine(),
            ]);
            throw $e;
        }
    }

    private function processScheduledOpportunities($now, &$processedCount): void
    {
        // Get opportunities that should be pushed now
        $opportunities = DB::table('opportunities')
            ->where('status', 'reviewed')
            ->where('score', '>=', 0.6)
            ->limit(50) // Process max 50 at a time
            ->get();

        if ($opportunities->isEmpty()) {
            return;
        }

        // Group opportunities by city for better targeting
        $groupedOpportunities = $opportunities->groupBy('city_id');

        foreach ($groupedOpportunities as $cityId => $cityOpportunities) {
            $opportunityIds = $cityOpportunities->pluck('id')->toArray();

            // Create segment based on city
            $segment = [
                'cities' => [$cityId],
                'notif_levels' => ['normal', 'high']
            ];

            // Dispatch push job
            \App\Jobs\PushOpportunitiesJob::dispatch($opportunityIds, $segment);
            $processedCount += count($opportunityIds);
        }

        Log::info('PushSchedulerJob: Processed opportunities', [
            'count' => $opportunities->count(),
            'cities' => $groupedOpportunities->keys()->toArray()
        ]);
    }

    private function processScheduledFeaturedAds($now, &$processedCount): void
    {
        // Get featured ads that should be pushed now
        $featuredAds = DB::table('featured_ads')
            ->where('status', 'scheduled')
            ->where('scheduled_at', '<=', $now)
            ->limit(20) // Process max 20 at a time
            ->get();

        if ($featuredAds->isEmpty()) {
            return;
        }

        foreach ($featuredAds as $featuredAd) {
            // Dispatch push job
            \App\Jobs\PushFeaturedAdsJob::dispatch($featuredAd->id);
            $processedCount++;
        }

        Log::info('PushSchedulerJob: Processed featured ads', [
            'count' => $featuredAds->count()
        ]);
    }
}
