<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class UserNotificationSetting extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'broadcasts_enabled',
        'announcements_enabled',
        'marketing_enabled',
        'preferences',
        'opted_out_at',
    ];

    protected $casts = [
        'broadcasts_enabled' => 'boolean',
        'announcements_enabled' => 'boolean',
        'marketing_enabled' => 'boolean',
        'preferences' => 'array',
        'opted_out_at' => 'datetime',
    ];

    /**
     * Get the user that owns the notification settings
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Scope for users who accept broadcasts
     */
    public function scopeBroadcastsEnabled($query)
    {
        return $query->where('broadcasts_enabled', true)
                    ->whereNull('opted_out_at');
    }

    /**
     * Scope for users who accept announcements
     */
    public function scopeAnnouncementsEnabled($query)
    {
        return $query->where('announcements_enabled', true)
                    ->whereNull('opted_out_at');
    }

    /**
     * Scope for users who accept marketing
     */
    public function scopeMarketingEnabled($query)
    {
        return $query->where('marketing_enabled', true)
                    ->whereNull('opted_out_at');
    }

    /**
     * Check if user accepts broadcasts
     */
    public function acceptsBroadcasts(): bool
    {
        return $this->broadcasts_enabled && is_null($this->opted_out_at);
    }

    /**
     * Check if user accepts announcements
     */
    public function acceptsAnnouncements(): bool
    {
        return $this->announcements_enabled && is_null($this->opted_out_at);
    }

    /**
     * Check if user accepts marketing messages
     */
    public function acceptsMarketing(): bool
    {
        return $this->marketing_enabled && is_null($this->opted_out_at);
    }

    /**
     * Opt out from all notifications
     */
    public function optOut(): void
    {
        $this->update([
            'opted_out_at' => now(),
        ]);
    }

    /**
     * Opt in to notifications
     */
    public function optIn(): void
    {
        $this->update([
            'opted_out_at' => null,
        ]);
    }

    /**
     * Check if user is opted out
     */
    public function isOptedOut(): bool
    {
        return !is_null($this->opted_out_at);
    }

    /**
     * Get or create notification settings for user
     */
    public static function forUser(User $user): self
    {
        return static::firstOrCreate(
            ['user_id' => $user->id],
            [
                'broadcasts_enabled' => true,
                'announcements_enabled' => true,
                'marketing_enabled' => true,
            ]
        );
    }
}