<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Create broadcast_campaigns table
        Schema::create('broadcast_campaigns', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->text('message');
            $table->json('attachments')->nullable(); // Images, files, etc.
            $table->json('inline_keyboard')->nullable(); // Telegram inline keyboard
            $table->enum('status', ['draft', 'scheduled', 'sending', 'sent', 'paused', 'cancelled'])
                  ->default('draft');
            $table->timestamp('scheduled_at')->nullable();
            $table->timestamp('started_at')->nullable();
            $table->timestamp('completed_at')->nullable();
            $table->integer('total_recipients')->default(0);
            $table->integer('sent_count')->default(0);
            $table->integer('failed_count')->default(0);
            $table->json('target_filters')->nullable(); // User filtering criteria
            $table->json('settings')->nullable(); // Additional settings
            $table->foreignId('created_by')->constrained('users')->onDelete('cascade');
            $table->timestamps();

            $table->index(['status', 'scheduled_at']);
            $table->index(['created_by']);
        });

        // Create broadcast_recipients table
        Schema::create('broadcast_recipients', function (Blueprint $table) {
            $table->id();
            $table->foreignId('campaign_id')->constrained('broadcast_campaigns')->onDelete('cascade');
            $table->foreignId('user_id')->constrained('users')->onDelete('cascade');
            $table->enum('status', ['pending', 'sent', 'failed', 'skipped'])->default('pending');
            $table->timestamp('sent_at')->nullable();
            $table->text('error_message')->nullable();
            $table->json('delivery_info')->nullable(); // Telegram message info
            $table->timestamps();

            $table->unique(['campaign_id', 'user_id']);
            $table->index(['campaign_id', 'status']);
            $table->index(['user_id']);
        });

        // Create notification_templates table
        Schema::create('notification_templates', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->string('key')->unique(); // Unique identifier for programmatic use
            $table->text('message');
            $table->json('attachments')->nullable();
            $table->json('inline_keyboard')->nullable();
            $table->json('variables')->nullable(); // Template variables
            $table->boolean('is_active')->default(true);
            $table->text('description')->nullable();
            $table->timestamps();

            $table->index(['key', 'is_active']);
        });

        // Create user_notification_settings table
        Schema::create('user_notification_settings', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained('users')->onDelete('cascade');
            $table->boolean('broadcasts_enabled')->default(true);
            $table->boolean('announcements_enabled')->default(true);
            $table->boolean('marketing_enabled')->default(true);
            $table->json('preferences')->nullable(); // Additional preferences
            $table->timestamp('opted_out_at')->nullable();
            $table->timestamps();

            $table->unique(['user_id']);
        });

        // Add notification preferences to users table
        Schema::table('users', function (Blueprint $table) {
            $table->timestamp('last_broadcast_at')->nullable()->after('last_activity_at');
            $table->boolean('notifications_enabled')->default(true)->after('last_broadcast_at');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('users', function (Blueprint $table) {
            $table->dropColumn(['last_broadcast_at', 'notifications_enabled']);
        });

        Schema::dropIfExists('user_notification_settings');
        Schema::dropIfExists('notification_templates');
        Schema::dropIfExists('broadcast_recipients');
        Schema::dropIfExists('broadcast_campaigns');
    }
};