# Configuration Guide

This guide covers all configuration options available in the LarBot Template framework for different hosting environments and use cases.

## Environment Configuration

### Development Environment (.env.local)

```env
# Application
APP_NAME="LarBot Template"
APP_ENV=local
APP_DEBUG=true
APP_URL=http://localhost:8000

# Database
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=larbot_dev
DB_USERNAME=root
DB_PASSWORD=

# Telegram Bot
TELEGRAM_BOT_TOKEN=your_dev_bot_token
TELEGRAM_BOT_USERNAME=your_dev_bot_username
TELEGRAM_WEBHOOK_URL=https://your-ngrok-url.ngrok.io/api/telegram/webhook

# Queue & Cache (Development)
QUEUE_CONNECTION=sync
CACHE_DRIVER=array
SESSION_DRIVER=file

# Mail (Development)
MAIL_MAILER=log

# Zarinpal (Sandbox)
ZARINPAL_MERCHANT_ID=your_sandbox_merchant_id
ZARINPAL_SANDBOX=true
```

### Production Environment (.env.production)

```env
# Application
APP_NAME="Your Bot Name"
APP_ENV=production
APP_DEBUG=false
APP_URL=https://yourdomain.com

# Database
DB_CONNECTION=mysql
DB_HOST=localhost
DB_PORT=3306
DB_DATABASE=production_db
DB_USERNAME=production_user
DB_PASSWORD=secure_password

# Telegram Bot
TELEGRAM_BOT_TOKEN=your_production_bot_token
TELEGRAM_BOT_USERNAME=your_production_bot_username
TELEGRAM_WEBHOOK_URL=https://yourdomain.com/api/telegram/webhook

# Queue & Cache (Production)
QUEUE_CONNECTION=database
CACHE_DRIVER=file
SESSION_DRIVER=file

# Mail (Production)
MAIL_MAILER=smtp
MAIL_HOST=mail.yourdomain.com
MAIL_PORT=587
MAIL_USERNAME=noreply@yourdomain.com
MAIL_PASSWORD=mail_password
MAIL_ENCRYPTION=tls

# Zarinpal (Production)
ZARINPAL_MERCHANT_ID=your_production_merchant_id
ZARINPAL_SANDBOX=false

# Security
APP_KEY=base64:generated_key_here
```

### cPanel Environment (.env.cpanel)

```env
# Application
APP_ENV=production
APP_DEBUG=false
APP_URL=https://yourdomain.com

# Database (cPanel format)
DB_CONNECTION=mysql
DB_HOST=localhost
DB_PORT=3306
DB_DATABASE=username_dbname
DB_USERNAME=username_dbuser
DB_PASSWORD=cpanel_db_password

# Optimized for shared hosting
QUEUE_CONNECTION=database
CACHE_DRIVER=file
SESSION_DRIVER=file
BROADCAST_DRIVER=log

# Disable unused services
TELESCOPE_ENABLED=false
HORIZON_ENABLED=false
```

## Module Configuration

### Payment Module (`config/modules/payment.php`)

```php
<?php

return [
    /*
    |--------------------------------------------------------------------------
    | Payment Module Configuration
    |--------------------------------------------------------------------------
    */

    'enabled' => env('PAYMENT_MODULE_ENABLED', true),

    'default_gateway' => 'zarinpal',

    'gateways' => [
        'zarinpal' => [
            'merchant_id' => env('ZARINPAL_MERCHANT_ID'),
            'sandbox_mode' => env('ZARINPAL_SANDBOX', false),
            'callback_url' => env('ZARINPAL_CALLBACK_URL', '/api/payment/zarinpal/callback'),
            'timeout' => 30, // seconds
            'description_prefix' => 'پرداخت در ',
        ],
    ],

    'currencies' => [
        'default' => 'IRR',
        'supported' => ['IRR', 'USD'],
        'conversion_rates' => [
            'USD_TO_IRR' => 42000, // Update regularly
        ],
    ],

    'subscription' => [
        'trial_days' => 7,
        'grace_period_days' => 3,
        'auto_renew' => true,
        'plans' => [
            'basic' => [
                'name' => 'پایه',
                'price_irr' => 50000,
                'duration_days' => 30,
                'features' => ['basic_features'],
            ],
            'premium' => [
                'name' => 'پریمیوم',
                'price_irr' => 100000,
                'duration_days' => 30,
                'features' => ['basic_features', 'premium_features'],
            ],
        ],
    ],

    'webhook' => [
        'verify_ssl' => true,
        'timeout' => 10,
        'retry_attempts' => 3,
    ],
];
```

### Notification Module (`config/modules/notification.php`)

```php
<?php

return [
    /*
    |--------------------------------------------------------------------------
    | Notification Module Configuration
    |--------------------------------------------------------------------------
    */

    'enabled' => env('NOTIFICATION_MODULE_ENABLED', true),

    'broadcast' => [
        'rate_limit' => [
            'messages_per_second' => 20,
            'messages_per_minute' => 100,
            'cooldown_hours' => 24, // Minimum hours between broadcasts to same user
        ],

        'batch_size' => 50, // Messages per batch job
        'retry_attempts' => 3,
        'retry_delay' => 300, // seconds

        'targeting' => [
            'max_recipients' => 10000,
            'allow_all_users' => false,
            'require_opt_in' => true,
        ],
    ],

    'templates' => [
        'max_variables' => 10,
        'variable_pattern' => '/\{\{(\w+)\}\}/',
        'default_variables' => [
            'app_name' => env('APP_NAME', 'LarBot'),
            'support_url' => env('SUPPORT_URL', '#'),
        ],
    ],

    'user_preferences' => [
        'default_opt_in' => true,
        'categories' => [
            'broadcasts' => [
                'name' => 'پیام‌های اطلاع‌رسانی',
                'enabled_by_default' => true,
            ],
            'announcements' => [
                'name' => 'اعلانات مهم',
                'enabled_by_default' => true,
            ],
            'marketing' => [
                'name' => 'پیام‌های تبلیغاتی',
                'enabled_by_default' => false,
            ],
        ],
    ],
];
```

### Analytics Module (`config/modules/analytics.php`)

```php
<?php

return [
    /*
    |--------------------------------------------------------------------------
    | Analytics Module Configuration
    |--------------------------------------------------------------------------
    */

    'enabled' => env('ANALYTICS_MODULE_ENABLED', true),

    'tracking' => [
        'auto_track_messages' => true,
        'auto_track_callbacks' => true,
        'track_user_agents' => true,
        'anonymize_ip' => true,
    ],

    'events' => [
        'session_timeout' => 1800, // 30 minutes
        'batch_size' => 100,
        'retention_days' => 365,
    ],

    'dashboard' => [
        'cache_duration' => 300, // 5 minutes
        'real_time_updates' => false,
        'export_formats' => ['csv', 'json'],
    ],

    'privacy' => [
        'anonymize_after_days' => 90,
        'delete_after_days' => 730, // 2 years
        'gdpr_compliant' => true,
    ],

    'performance' => [
        'enable_query_caching' => true,
        'optimize_for_large_datasets' => true,
        'use_database_indexes' => true,
    ],
];
```

### Persian Module (`config/modules/persian.php`)

```php
<?php

return [
    /*
    |--------------------------------------------------------------------------
    | Persian Localization Configuration
    |--------------------------------------------------------------------------
    */

    'enabled' => env('PERSIAN_MODULE_ENABLED', true),

    'locale' => [
        'default' => 'fa',
        'fallback' => 'en',
        'supported' => ['fa', 'en'],
    ],

    'calendar' => [
        'type' => 'jalali', // jalali, hijri, gregorian
        'format' => 'Y/m/d',
        'show_weekday' => true,
    ],

    'numbers' => [
        'auto_convert' => true,
        'convert_dates' => true,
        'convert_currency' => true,
        'persian_digits' => true,
    ],

    'text' => [
        'rtl_support' => true,
        'auto_direction' => true,
        'text_alignment' => 'right',
    ],

    'cultural' => [
        'formal_address' => true, // Use formal Persian pronouns
        'religious_calendar' => true,
        'local_holidays' => true,
    ],
];
```

### Content Module (`config/modules/content.php`)

```php
<?php

return [
    /*
    |--------------------------------------------------------------------------
    | Content Management Configuration
    |--------------------------------------------------------------------------
    */

    'enabled' => env('CONTENT_MODULE_ENABLED', true),

    'pages' => [
        'cache_duration' => 3600, // 1 hour
        'auto_publish' => false,
        'version_control' => true,
        'seo_optimization' => true,
    ],

    'menus' => [
        'max_depth' => 3,
        'cache_duration' => 1800, // 30 minutes
        'auto_sort' => true,
        'responsive' => true,
    ],

    'media' => [
        'max_file_size' => 10485760, // 10MB
        'allowed_types' => ['jpg', 'jpeg', 'png', 'gif', 'pdf', 'doc', 'docx'],
        'storage_disk' => 'public',
        'optimize_images' => true,
    ],

    'editor' => [
        'toolbar' => 'full', // minimal, basic, full
        'autosave' => true,
        'word_count' => true,
        'spell_check' => true,
    ],
];
```

## System Configuration

### Core Framework (`config/larbot.php`)

```php
<?php

return [
    /*
    |--------------------------------------------------------------------------
    | LarBot Framework Configuration
    |--------------------------------------------------------------------------
    */

    'version' => '1.0.0',

    'modules' => [
        'auto_discovery' => true,
        'cache_registration' => true,
        'load_order' => [
            'persian',
            'analytics',
            'content',
            'payment',
            'notification',
        ],
    ],

    'telegram' => [
        'api_url' => 'https://api.telegram.org',
        'timeout' => 30,
        'retry_attempts' => 3,
        'rate_limit' => [
            'messages_per_second' => 30,
            'max_concurrent' => 10,
        ],
    ],

    'security' => [
        'webhook_secret' => env('TELEGRAM_WEBHOOK_SECRET'),
        'ip_whitelist' => [
            '149.154.160.0/20',
            '91.108.4.0/22',
        ],
        'csrf_protection' => true,
        'rate_limiting' => true,
    ],

    'performance' => [
        'cache_responses' => true,
        'optimize_queries' => true,
        'lazy_load_modules' => true,
        'compress_responses' => true,
    ],
];
```

### Database Configuration (`config/database.php` additions)

```php
// Add to existing database.php
'connections' => [
    // ... existing connections

    'analytics' => [
        'driver' => 'mysql',
        'url' => env('ANALYTICS_DATABASE_URL'),
        'host' => env('ANALYTICS_DB_HOST', env('DB_HOST', '127.0.0.1')),
        'port' => env('ANALYTICS_DB_PORT', env('DB_PORT', '3306')),
        'database' => env('ANALYTICS_DB_DATABASE', env('DB_DATABASE', 'forge')),
        'username' => env('ANALYTICS_DB_USERNAME', env('DB_USERNAME', 'forge')),
        'password' => env('ANALYTICS_DB_PASSWORD', env('DB_PASSWORD', '')),
        // ... other settings
    ],
],

'redis' => [
    'options' => [
        'cluster' => env('REDIS_CLUSTER', 'redis'),
        'prefix' => env('REDIS_PREFIX', Str::slug(env('APP_NAME', 'laravel'), '_').'_database_'),
    ],

    'cache' => [
        'url' => env('REDIS_URL'),
        'host' => env('REDIS_HOST', '127.0.0.1'),
        'password' => env('REDIS_PASSWORD', null),
        'port' => env('REDIS_PORT', '6379'),
        'database' => env('REDIS_CACHE_DB', '1'),
    ],

    'notifications' => [
        'url' => env('REDIS_URL'),
        'host' => env('REDIS_HOST', '127.0.0.1'),
        'password' => env('REDIS_PASSWORD', null),
        'port' => env('REDIS_PORT', '6379'),
        'database' => env('REDIS_NOTIFICATION_DB', '2'),
    ],
],
```

### Queue Configuration (`config/queue.php` additions)

```php
'connections' => [
    // ... existing connections

    'notifications' => [
        'driver' => 'database',
        'table' => 'notification_jobs',
        'queue' => 'notifications',
        'retry_after' => 90,
        'after_commit' => false,
    ],

    'analytics' => [
        'driver' => 'database',
        'table' => 'analytics_jobs',
        'queue' => 'analytics',
        'retry_after' => 60,
        'after_commit' => false,
    ],
],

'failed' => [
    'driver' => env('QUEUE_FAILED_DRIVER', 'database'),
    'database' => env('DB_CONNECTION', 'mysql'),
    'table' => 'failed_jobs',
],
```

## Hosting-Specific Configurations

### Shared Hosting (cPanel)

```php
// config/cpanel.php
return [
    'optimization' => [
        'opcache' => [
            'enable' => true,
            'memory_consumption' => 128,
            'max_accelerated_files' => 10000,
            'revalidate_freq' => 2,
        ],

        'laravel' => [
            'config_cache' => true,
            'route_cache' => true,
            'view_cache' => true,
            'event_cache' => true,
        ],

        'database' => [
            'persistent_connections' => false,
            'query_cache' => true,
            'slow_query_log' => false,
        ],
    ],

    'limits' => [
        'memory_limit' => '512M',
        'max_execution_time' => 30,
        'upload_max_filesize' => '10M',
        'post_max_size' => '10M',
    ],

    'security' => [
        'hide_laravel_headers' => true,
        'disable_server_signature' => true,
        'force_https' => true,
    ],
];
```

### VPS/Dedicated Server

```php
// config/server.php
return [
    'optimization' => [
        'redis' => [
            'enabled' => true,
            'cache_driver' => 'redis',
            'session_driver' => 'redis',
            'queue_driver' => 'redis',
        ],

        'supervisor' => [
            'enabled' => true,
            'workers' => 4,
            'memory_limit' => 128,
        ],

        'nginx' => [
            'gzip' => true,
            'static_cache' => true,
            'rate_limiting' => true,
        ],
    ],

    'monitoring' => [
        'logs' => [
            'level' => 'error',
            'rotate' => true,
            'max_files' => 30,
        ],

        'metrics' => [
            'enabled' => true,
            'collect_system_metrics' => true,
            'performance_monitoring' => true,
        ],
    ],
];
```

## Environment-Specific Settings

### Development Settings

```php
// config/development.php
return [
    'debug' => [
        'queries' => true,
        'routes' => true,
        'events' => true,
        'telegram_logs' => true,
    ],

    'testing' => [
        'mock_telegram_api' => true,
        'mock_payment_gateway' => true,
        'fake_notifications' => true,
    ],

    'tools' => [
        'telescope' => true,
        'horizon' => false,
        'debugbar' => true,
    ],
];
```

### Staging Settings

```php
// config/staging.php
return [
    'debug' => [
        'queries' => false,
        'routes' => false,
        'events' => false,
        'telegram_logs' => true,
    ],

    'testing' => [
        'mock_telegram_api' => false,
        'mock_payment_gateway' => true,
        'fake_notifications' => false,
    ],

    'monitoring' => [
        'log_level' => 'info',
        'error_reporting' => true,
        'performance_tracking' => true,
    ],
];
```

### Production Settings

```php
// config/production.php
return [
    'debug' => [
        'queries' => false,
        'routes' => false,
        'events' => false,
        'telegram_logs' => false,
    ],

    'security' => [
        'force_https' => true,
        'hsts_max_age' => 31536000,
        'content_security_policy' => true,
        'xss_protection' => true,
    ],

    'performance' => [
        'cache_everything' => true,
        'minify_responses' => true,
        'compress_assets' => true,
        'cdn_enabled' => false,
    ],

    'monitoring' => [
        'log_level' => 'error',
        'error_reporting' => true,
        'performance_tracking' => true,
        'uptime_monitoring' => true,
    ],
];
```

## Advanced Configuration

### Custom Module Configuration

```php
// config/modules/custom.php
return [
    'enabled' => env('CUSTOM_MODULE_ENABLED', false),
    
    'settings' => [
        'api_key' => env('CUSTOM_API_KEY'),
        'api_url' => env('CUSTOM_API_URL', 'https://api.example.com'),
        'timeout' => env('CUSTOM_TIMEOUT', 30),
    ],

    'features' => [
        'feature_a' => env('CUSTOM_FEATURE_A', true),
        'feature_b' => env('CUSTOM_FEATURE_B', false),
    ],

    'limits' => [
        'requests_per_minute' => 60,
        'max_payload_size' => 1048576, // 1MB
    ],
];
```

### Multi-Bot Configuration

```php
// config/bots.php
return [
    'default' => env('DEFAULT_BOT', 'main'),

    'bots' => [
        'main' => [
            'token' => env('MAIN_BOT_TOKEN'),
            'username' => env('MAIN_BOT_USERNAME'),
            'webhook_url' => env('MAIN_BOT_WEBHOOK_URL'),
            'modules' => ['persian', 'payment', 'analytics'],
        ],

        'support' => [
            'token' => env('SUPPORT_BOT_TOKEN'),
            'username' => env('SUPPORT_BOT_USERNAME'),
            'webhook_url' => env('SUPPORT_BOT_WEBHOOK_URL'),
            'modules' => ['persian', 'content'],
        ],
    ],
];
```

## Configuration Validation

### Environment Validation

```php
// config/validation.php
return [
    'required_env_vars' => [
        'APP_KEY',
        'DB_DATABASE',
        'DB_USERNAME',
        'TELEGRAM_BOT_TOKEN',
    ],

    'conditional_requirements' => [
        'ZARINPAL_MERCHANT_ID' => 'PAYMENT_MODULE_ENABLED',
        'REDIS_HOST' => 'CACHE_DRIVER=redis',
        'MAIL_HOST' => 'NOTIFICATION_MODULE_ENABLED',
    ],

    'format_validation' => [
        'TELEGRAM_BOT_TOKEN' => '/^\d+:[A-Za-z0-9_-]{35}$/',
        'ZARINPAL_MERCHANT_ID' => '/^[a-f0-9-]{36}$/',
    ],
];
```

This comprehensive configuration guide covers all aspects of configuring the LarBot Template for different environments and use cases. Each configuration option is documented with its purpose and recommended values for various hosting scenarios.