# cPanel Installation Guide

This guide will walk you through deploying the LarBot Template on cPanel shared hosting, optimized for Iranian hosting providers and Persian language support.

## Prerequisites

- **cPanel hosting account** with the following minimum requirements:
  - PHP 8.3+ (with required extensions)
  - MySQL 8.0+ or MariaDB 10.3+
  - At least 512MB RAM (1GB+ recommended)
  - Composer access (via SSH or terminal)
  - Cron job support
  - SSL certificate (Let's Encrypt recommended)

## Step 1: Prepare Your Local Environment

### 1.1 Download and Configure
```bash
# Clone the template
git clone https://github.com/your-username/larbot-template.git
cd larbot-template

# Install dependencies locally
composer install --no-dev --optimize-autoloader
npm install && npm run build

# Configure environment
cp .env.example .env.cpanel
```

### 1.2 Configure .env.cpanel
```env
# Application
APP_NAME="Your Bot Name"
APP_ENV=production
APP_DEBUG=false
APP_URL=https://yourdomain.com

# Database (get from cPanel MySQL section)
DB_CONNECTION=mysql
DB_HOST=localhost
DB_PORT=3306
DB_DATABASE=username_botdb
DB_USERNAME=username_botuser
DB_PASSWORD=your_database_password

# Telegram Bot Configuration
TELEGRAM_BOT_TOKEN=your_bot_token_from_botfather
TELEGRAM_BOT_USERNAME=your_bot_username
TELEGRAM_WEBHOOK_URL=https://yourdomain.com/api/telegram/webhook

# Zarinpal (Iranian Payment Gateway)
ZARINPAL_MERCHANT_ID=your_zarinpal_merchant_id
ZARINPAL_SANDBOX=false

# Queue Configuration (Database driver for shared hosting)
QUEUE_CONNECTION=database
CACHE_DRIVER=file
SESSION_DRIVER=file

# Mail Configuration (if needed)
MAIL_MAILER=smtp
MAIL_HOST=mail.yourdomain.com
MAIL_PORT=587
MAIL_USERNAME=noreply@yourdomain.com
MAIL_PASSWORD=your_email_password
MAIL_ENCRYPTION=tls

# Security
APP_KEY=
```

### 1.3 Generate Application Key
```bash
php artisan key:generate --env=production
```

## Step 2: Prepare Files for Upload

### 2.1 Create Deployment Package
```bash
# Remove development files
rm -rf node_modules tests .git .github
rm .env.example .gitignore phpunit.xml

# Optimize Laravel for production
php artisan config:cache
php artisan route:cache
php artisan view:cache

# Create upload package
cd ..
zip -r larbot-template.zip larbot-template/
```

## Step 3: cPanel Database Setup

### 3.1 Create Database
1. Log into your cPanel
2. Go to **MySQL Databases**
3. Create a new database: `username_botdb`
4. Create a database user: `username_botuser`
5. Set a strong password
6. Add user to database with **ALL PRIVILEGES**

### 3.2 Note Database Credentials
```
Database Host: localhost
Database Name: username_botdb
Database User: username_botuser
Database Password: [your chosen password]
```

## Step 4: File Upload and Directory Structure

### 4.1 Upload Files
1. Upload `larbot-template.zip` to your cPanel File Manager
2. Extract the zip file in your account root (not public_html)
3. You should have this structure:
```
/home/username/
├── public_html/
├── larbot-template/
│   ├── app/
│   ├── bootstrap/
│   ├── config/
│   ├── database/
│   ├── public/
│   ├── resources/
│   ├── routes/
│   ├── storage/
│   └── vendor/
└── logs/
```

### 4.2 Configure Public Directory
Since cPanel serves from `public_html`, you need to link Laravel's public directory:

**Option A: Symlink (if supported)**
```bash
# Via cPanel Terminal (if available)
cd /home/username/
rm -rf public_html
ln -s larbot-template/public public_html
```

**Option B: Copy files (universal)**
1. Copy contents of `larbot-template/public/` to `public_html/`
2. Edit `public_html/index.php`:
```php
<?php
// Change these paths to point to your Laravel installation
require __DIR__.'/../larbot-template/vendor/autoload.php';
$app = require_once __DIR__.'/../larbot-template/bootstrap/app.php';
```

### 4.3 Set Directory Permissions
```bash
# Storage and cache directories need write permissions
chmod -R 775 larbot-template/storage
chmod -R 775 larbot-template/bootstrap/cache

# If using symlink
chmod -R 755 public_html
```

## Step 5: Environment Configuration

### 5.1 Upload Environment File
1. Rename `.env.cpanel` to `.env`
2. Upload `.env` to `/home/username/larbot-template/`
3. Ensure the file is not accessible via web (should be outside public_html)

### 5.2 Configure Storage
```bash
# Create required storage directories
mkdir -p larbot-template/storage/logs
mkdir -p larbot-template/storage/framework/cache
mkdir -p larbot-template/storage/framework/sessions
mkdir -p larbot-template/storage/framework/views
mkdir -p larbot-template/storage/app/public

# Set permissions
chmod -R 775 larbot-template/storage
```

## Step 6: Database Migration

### 6.1 Run Migrations via cPanel Terminal
If your hosting provider supports terminal access:
```bash
cd /home/username/larbot-template
php artisan migrate --force
php artisan db:seed --force
```

### 6.2 Alternative: Manual SQL Import
If terminal access is not available:
1. Export migration SQL locally:
```bash
php artisan schema:dump --database=mysql
```
2. Import the generated SQL file via cPanel phpMyAdmin

## Step 7: Telegram Bot Setup

### 7.1 Create Telegram Bot
1. Message [@BotFather](https://t.me/botfather) on Telegram
2. Use `/newbot` command
3. Follow instructions to get bot token
4. Update `.env` file with token and username

### 7.2 Set Webhook
Via cPanel Terminal (if available):
```bash
cd /home/username/larbot-template
php artisan telegram:setup
```

Or manually via browser:
```
https://api.telegram.org/bot{BOT_TOKEN}/setWebhook?url=https://yourdomain.com/api/telegram/webhook
```

## Step 8: Configure Cron Jobs

### 8.1 Laravel Scheduler
Add this cron job in cPanel **Cron Jobs** section:
```bash
# Laravel Task Scheduler (runs every minute)
* * * * * cd /home/username/larbot-template && php artisan schedule:run >> /dev/null 2>&1
```

### 8.2 Queue Worker
For background job processing:
```bash
# Queue worker (runs every 5 minutes)
*/5 * * * * cd /home/username/larbot-template && php artisan queue:work --stop-when-empty --max-time=240 >> /dev/null 2>&1
```

### 8.3 Log Cleanup
```bash
# Clean old logs (weekly)
0 2 * * 0 cd /home/username/larbot-template && php artisan log:clear --days=30 >> /dev/null 2>&1
```

## Step 9: SSL and Security Configuration

### 9.1 Enable SSL
1. In cPanel, go to **SSL/TLS**
2. Enable **Let's Encrypt** for your domain
3. Force HTTPS redirects

### 9.2 Security Headers
Add to your `.htaccess` in `public_html`:
```apache
# Security Headers
Header always set X-Content-Type-Options nosniff
Header always set X-Frame-Options DENY
Header always set X-XSS-Protection "1; mode=block"
Header always set Strict-Transport-Security "max-age=31536000; includeSubDomains"

# Hide Laravel
Header unset X-Powered-By
ServerSignature Off
```

## Step 10: Testing and Verification

### 10.1 Health Check
Visit: `https://yourdomain.com/admin/health`
- All indicators should be green
- Database connectivity confirmed
- File permissions correct

### 10.2 Admin Panel Access
1. Visit: `https://yourdomain.com/admin`
2. Default credentials (change immediately):
   - Email: `admin@example.com`
   - Password: `password`

### 10.3 Bot Testing
1. Send `/start` to your bot on Telegram
2. Test basic commands: `/help`, `/settings`
3. Check admin panel for user registration

## Step 11: Production Optimizations

### 11.1 Performance Tuning
```bash
# Clear and optimize caches
php artisan config:cache
php artisan route:cache
php artisan view:cache
php artisan event:cache

# Optimize Composer autoloader
composer dump-autoload --optimize --no-dev
```

### 11.2 Error Monitoring
Set up log monitoring:
```bash
# Add to cron for error alerts
0 */6 * * * cd /home/username/larbot-template && php artisan log:check --email=admin@yourdomain.com
```

## Troubleshooting Common Issues

### Issue 1: 500 Internal Server Error
**Causes:**
- Incorrect file permissions
- Missing .env file
- Database connection issues

**Solutions:**
```bash
# Check permissions
chmod -R 775 storage bootstrap/cache

# Verify .env file exists
ls -la larbot-template/.env

# Test database connection
php artisan tinker
>>> DB::connection()->getPdo();
```

### Issue 2: Webhook Not Working
**Causes:**
- Invalid SSL certificate
- Incorrect webhook URL
- Server not accepting POST requests

**Solutions:**
```bash
# Check webhook status
curl -X GET "https://api.telegram.org/bot{BOT_TOKEN}/getWebhookInfo"

# Reset webhook
curl -X POST "https://api.telegram.org/bot{BOT_TOKEN}/deleteWebhook"
curl -X POST "https://api.telegram.org/bot{BOT_TOKEN}/setWebhook?url=https://yourdomain.com/api/telegram/webhook"
```

### Issue 3: Queue Jobs Not Processing
**Causes:**
- Cron job not configured
- Database queue table missing
- Insufficient memory

**Solutions:**
```bash
# Check queue status
php artisan queue:failed

# Process stuck jobs
php artisan queue:retry all

# Restart queue worker
php artisan queue:restart
```

### Issue 4: Storage Permission Issues
**Solution:**
```bash
# Fix storage permissions recursively
find larbot-template/storage -type f -exec chmod 664 {} \;
find larbot-template/storage -type d -exec chmod 775 {} \;
```

## Performance Optimization for cPanel

### 1. Enable OPcache
Add to `.htaccess` or contact hosting provider:
```apache
# Enable OPcache
php_value opcache.enable 1
php_value opcache.memory_consumption 128
php_value opcache.max_accelerated_files 10000
```

### 2. Memory Optimization
In `.env`:
```env
# Optimize for shared hosting
CACHE_DRIVER=file
SESSION_DRIVER=file
QUEUE_CONNECTION=database
BROADCAST_DRIVER=log
```

### 3. Database Optimization
```sql
-- Add indexes for better performance
ALTER TABLE users ADD INDEX idx_telegram_id (telegram_id);
ALTER TABLE analytics_events ADD INDEX idx_user_created (user_id, created_at);
ALTER TABLE broadcast_recipients ADD INDEX idx_campaign_status (campaign_id, status);
```

## Maintenance Schedule

### Daily
- Monitor error logs
- Check bot responsiveness
- Verify webhook status

### Weekly
- Review analytics data
- Clean old cache files
- Update content as needed

### Monthly
- Security updates
- Database optimization
- Performance review
- Backup verification

## Support and Updates

### Getting Help
1. Check logs: `/home/username/larbot-template/storage/logs/`
2. Admin panel health check: `https://yourdomain.com/admin/health`
3. Community support: GitHub Issues

### Updates
1. Backup current installation
2. Download latest version
3. Merge configuration changes
4. Test in staging environment
5. Deploy to production

---

**Note**: This guide assumes a typical cPanel shared hosting environment. Some steps may vary depending on your hosting provider. Always consult your hosting provider's documentation for specific configurations.