# RUNBOOK

## Table of Contents
- Prerequisites
- Setup & Configuration
- Database: Migrate & Seed
- Running Queues & Schedule
- Endpoints & Simulations
- Testing Feature Guards & Quotas
- Troubleshooting

## Prerequisites
- PHP 8.3+
- MySQL 8+
- Composer 2.x
- Redis or Database queue (per environment)

Related docs: see `docs/LOGGING_POLICY.md` (logging and alerts) and `docs/DOCUMENTATION_GUIDE.md` (how to keep docs in sync with code).

## Setup & Configuration
1) Clone repo and install dependencies:
   - `composer install`
2) Copy env and configure:
   - `cp .env.example .env`
- Set `APP_URL`, DB credentials, queue driver, and:
     - `TELEGRAPH_TOKEN`, `TELEGRAPH_SECRET`
     - `OPENAI_API_KEY` (for Responses API)
     - `ZARINPAL_MERCHANT_ID`, `ZARINPAL_CALLBACK_URL`
     - Growth caps: `GROWTH_TRIAL_PENALTY`, `GROWTH_CHALLENGE_REWARD`, `GROWTH_REFERRAL_DAILY_CAP`
3) Generate key:
   - `php artisan key:generate`

### Local Setup (Developer)
- Clone + install: `composer install`
- Env: `cp .env.example .env` then set `APP_URL=http://127.0.0.1:8000`, DB creds, and required keys
- Migrate/seed: `php artisan migrate --seed`
- Queue tables: `php artisan queue:table && php artisan migrate`
- Serve: `php artisan serve` (or `php -S 127.0.0.1:8000 -t public`)
- Worker: `php artisan queue:work --queue=default --tries=3 --backoff=5`
- Register telegraph routes (see below) and set webhook (replace APP_URL with local tunnel if needed)

### Production Setup (Laravel CPanel Hosting)
**Server Environment**: Laravel CPanel hosting with the following characteristics:
- **PHP Version**: 8.3+ (confirmed compatible)
- **Extensions**: ext-intl enabled (required for Filament)
- **Queue System**: Database-based queues (CPanel hosting limitation)
- **Cron Jobs**: Available through CPanel interface
- **File Permissions**: Standard shared hosting permissions
- **SSL**: Automatic SSL certificates available

**Production Deployment Steps**:
1. **File Upload**: Upload all project files via File Manager or FTP
2. **Environment Configuration**:
   - Copy `.env.example` to `.env`
   - Set `APP_URL` to your domain (e.g., `https://yourdomain.com`)
   - Configure database credentials from CPanel
   - Set `QUEUE_CONNECTION=database`
   - Add Telegram bot credentials: `TELEGRAPH_TOKEN` and `TELEGRAPH_SECRET`
3. **Database Setup**:
   - Create database through CPanel
   - Run: `php artisan migrate --seed`
   - Run: `php artisan queue:table && php artisan migrate`
4. **Cron Job Setup** (via CPanel Cron Jobs):
   - Add: `* * * * * /usr/local/bin/php /home/username/public_html/artisan schedule:run`
   - Frequency: Every minute
5. **Queue Worker** (Background Process):
   - CPanel hosting typically doesn't support persistent workers
   - Alternative: Use cron-based queue processing every minute
   - Add: `* * * * * /usr/local/bin/php /home/username/public_html/artisan queue:work --stop-when-empty`
6. **Telegram Webhook Setup**:
   - Set webhook URL to: `https://yourdomain.com/api/telegram/webhook`
   - Include secret token for security
7. **Cache and Route Optimization**:
   - Run: `php artisan config:cache`
   - Run: `php artisan route:cache`
   - Run: `php artisan view:cache`

**CPanel-Specific Considerations**:
- File permissions: Ensure `storage/` and `bootstrap/cache/` are writable (755/644)
- PHP memory limit: Verify sufficient memory allocation (512MB+ recommended)
- Execution time: Check max_execution_time for long-running operations
- Database connections: Monitor connection limits on shared hosting

## Database: Migrate & Seed
- Run migrations: `php artisan migrate`
- Seed initial data:
  - `php artisan db:seed --class=Database\Seeders\DatabaseSeeder`
  - Seeds: Plans, Plan Entitlements, Cities/Aliases, Menu, Prompt Configs, Checklists, Experiments.

## Running Queues & Schedule
- Queues: start worker
  - `php artisan queue:work --queue=default --tries=3 --backoff=5`
- Schedule: ensure app/Console/Kernel.php schedules jobs (add if missing)
  - PushSchedulerJob (hourly)
  - ScoreOpportunitiesJob (*/15)
  - GrowthDailyJob (daily at 02:00)
  - PaymentsReconcileJob (hourly)
- Cron on server (cPanel):
  - `* * * * * php /path/to/artisan schedule:run >> /dev/null 2>&1`
  - Schedules registered in `app/Console/Kernel.php`:
    - PushSchedulerJob at 09, 11, 14 (`0 9,11,14 * * *`)
    - ScoreOpportunitiesJob every 30 minutes
    - GrowthDailyJob at 02:00
    - PaymentsReconcileJob hourly

## Telegram Bot Setup

### Webhook Configuration
- Ensure `routes/telegraph.php` is registered (RouteServiceProvider or `require` in routes/api.php).
- **Manual Setup Commands** (replace with actual values):

```bash
# Set webhook (single line)
curl -X POST "https://api.telegram.org/bot7382889700:AAHYWSXHbQKUunHLURpy742f4cbBjzAzZMo/setWebhook" -H "Content-Type: application/json" -d '{"url":"https://lar.hanafi.vip/api/telegram/webhook","secret_token":"f35f1bfbad29a33309bdc2b1c761d6e0"}'

# Verify webhook
curl "https://api.telegram.org/bot7382889700:AAHYWSXHbQKUunHLURpy742f4cbBjzAzZMo/getWebhookInfo"

# Test webhook endpoint
curl -i -X POST "https://lar.hanafi.vip/api/telegram/webhook" -H "X-Telegram-Bot-Api-Secret-Token: f35f1bfbad29a33309bdc2b1c761d6e0" -H "Content-Type: application/json" -d '{}'
```

- **Using Laravel Command** (after fixes):
```bash
php artisan telegram:setup
```

Note: This project uses Option A (API prefix). Effective path is `POST /api/telegram/webhook`.

### Common Issues & Solutions

#### 1. "Call to undefined method json()" Error
**Problem**: Telegraph response methods changed in newer versions.
**Solution**: Always call `->send()` before `->json()` on Telegraph responses.

#### 2. "Call to undefined method setWebhook()" Error  
**Problem**: Method name changed in Telegraph package.
**Solution**: Use `$bot->registerWebhook()->send()` instead of `Telegraph::bot($bot)->setWebhook()`.

#### 3. "No TelegraphBot defined for this request" Error
**Problem**: Bot not found in database or service initialization failed.
**Solutions**:
- Run `php artisan db:seed --class=TelegramBotSeeder`
- Check TELEGRAPH_TOKEN in .env
- Verify telegraph_bots table has records

#### 4. Foreign Key Constraint Violation (events table)
**Problem**: Trying to create events for non-existent users.
**Solution**: Ensure user creation happens before event tracking:
```php
// In webhook handler, create user first
$user = User::updateOrCreate(
    ['tg_id' => $userId],
    ['name' => $firstName, 'role' => 'buyer'] // defaults
);
// Then track events
EventBus::track($user->id, 'Start', ['chat_id' => $chatId]);
```

#### 5. "Call to undefined method caption()" Error (RESOLVED)
**Problem**: `Call to undefined method DefStudio\Telegraph\Telegraph::caption()` when sending documents with captions.
**Root Cause**: The `caption()` method is not available in the DefStudio Telegraph library.
**Solution Applied**: 
- **File**: `app/Services/Messaging/TelegramBot.php` (line 493)
- **Fix**: Replace `->caption($caption)` with `->withData('caption', $caption)`
- **Code Change**:
```php
// Before (INCORRECT)
$telegraph->document($filePath)->caption($this->ensureUtf8($caption))->send();

// After (CORRECT)
$telegraph->document($filePath)->withData('caption', $this->ensureUtf8($caption))->send();
```
**Status**: ✅ Fixed - Document sending with captions now works correctly

#### 6. Contract Files Not Found Error (RESOLVED)
**Problem**: Contract files missing when using `/قولنامه` command - files were in `contracts/` but code looked in `storage/app/contracts/`.
**Root Cause**: File path mismatch between actual file location and expected location in `sendContractFiles` method.
**Solution Applied**:
- **Action**: Copied contract files from `contracts/` to `storage/app/contracts/` directory
- **Files Available**:
  - `sample_contract.pdf` (744 bytes) - Main contract file
  - `buy_contract_template.txt` (968 bytes) - Buy contract template  
  - `sell_contract_template.txt` (968 bytes) - Sell contract template
  - `sample_contract.txt` (1517 bytes) - Additional sample contract
- **Verification Command**: `Get-ChildItem storage/app/contracts/ -Force` (PowerShell)
**Status**: ✅ Fixed - All contract files are now accessible and can be sent via Telegram

#### 7. Legal Disclaimer Message Inconsistency (RESOLVED)
**Problem**: Inconsistent and potentially misleading legal disclaimer messages in contract delivery.
**Root Cause**: Hardcoded disclaimer messages not using the centralized constant.
**Solution Applied**:
- **File**: `app/Http/Controllers/TelegramWebhookController.php`
- **Constant Updated**: `LEGAL_DISCLAIMER` changed from warning message to professional message
- **New Message**: "📋 این قولنامه توسط یک وکیل مجرب نوشته شده است. با پرینت گرفتن می‌توانید استفاده کنید."
- **Consistency Fix**: All messages in `sendContractFiles` method now use `self::LEGAL_DISCLAIMER` constant
**Status**: ✅ Fixed - Unified, professional legal messaging across all contract communications

## Production Setup (cPanel Checklist)
- PHP version: 8.2; enable intl extension in PHP Selector.
- Docroot: point domain to project `public/` directory.
- Permissions: `storage/` and `bootstrap/cache/` writable by PHP user.
- Composer install: run in project root; optimize (optional) `php artisan optimize`.
- Env: set `.env` with real keys (Telegram token/secret, Zarinpal merchant, OpenAI, MajidAPI); secure file permissions.
- DB: create database/user; import/migrate: `php artisan migrate --seed`.
- Queue: create tables `php artisan queue:table && php artisan migrate`; start worker via Cron or Supervisor (if available):
  - Cron example (every minute): `* * * * * php /path/to/artisan queue:work --stop-when-empty >> /dev/null 2>&1`
  - Prefer persistent worker (Supervisor) for reliability.
- Scheduler: Cron entry `* * * * * php /path/to/artisan schedule:run`.
- Telegraph routes: ensure registered; set webhook as above with production `APP_URL`.
- Health checks:
  - `php artisan route:list | grep -E "telegram|payments|user/saved_ads"`
  - `curl -i -X POST $APP_URL/api/telegram/webhook -H "X-Telegram-Bot-Api-Secret-Token: $TELEGRAPH_SECRET" -d '{}'` → 200
  - `curl "https://api.telegram.org/bot$TELEGRAPH_TOKEN/getWebhookInfo"` → url ok; no last_error
  - `tail -n 200 storage/logs/laravel.log` (errors)

## Health & Diagnostics
- Routes: `php artisan route:list`
- Cache clear: `php artisan optimize:clear`
- DB connectivity: `php artisan tinker` then DB::select('select 1')
- Queue: check `jobs` / `failed_jobs` tables; `php artisan queue:retry all` for retries
- Payments: verify logs in `payments` and `subscriptions`; check `status`, `ref`, dates
- Ads: check `ads_raw` and `opportunities` counts after running jobs


## Endpoints & Simulations
- Telegram webhook (Telegraph): `POST /api/telegram/webhook`
  - Configure webhook URL to `APP_URL/api/telegram/webhook`.
  - Validate secret header if implemented.
- AI valuation: `POST /api/ai/respond`
- Uses OpenAI Responses API with `tool_resources.file_search` support and `store:true`. Pass `response_format.type=json_object` for structured output (valuation schema).
 - PriceService normalizes input (digits, Jalali/Gregorian, stopwords) then queries MajidAPI `carPrice` + `search/latest`; AI summarization must remain informational (هیچ CTA تماس/خرید مجاز نیست).
- Payments:
  - Create: `POST /api/payments/create {"plan":"basic"}`
  - Verify (sandbox): `GET /api/payments/verify?authority=...&Status=OK` (idempotent by authority). Accepts case-insensitive query keys.
  - Reconcile: `POST /api/payments/reconcile`
- Saved Ads:
  - Save: `POST /api/user/saved_ads` (see API spec)
  - List: `GET /api/user/saved_ads`
  - Remove: `DELETE /api/user/saved_ads/{id}`
- Simulate push click:
  - `POST /api/simulate/click {"id": <push_logs.id>}`
  - Guarded by environment; set `APP_ENV=local` (SimulateProtect).
  - Sets `opened_at` (first time), `clicked_cta_at`, and emits `PushOpen`, `ViewDeal`, `CallFromCard` events.

## Local Fixture Ingestion (Ads)
- Place fixture JSON files at:
  - `storage/app/fixtures/bama_latest.json` (array of simplified ad items)
  - `storage/app/fixtures/divar_latest.json` (array of simplified ad items)
- Run ingestion + scoring:
  - `php artisan queue:work` to run `BamaPullLatestJob` or dispatch manually.
  - `php artisan tinker` → `App\Jobs\BamaPullLatestJob::dispatch();` then `App\Jobs\ScoreOpportunitiesJob::dispatch();`
  - Or call within a route/command as needed.

## Testing Feature Guards & Quotas
- AI route is limited to 5/min/user: exercise `/api/ai/respond`
  - Expect 429 with `Retry-After` header once limit exceeded (app/Http/Middleware/RateLimitPerUser.php:20–26).
- FeatureGuard examples:
  - `feature.guard:pricecheck` on `/api/ai/respond` consumes `valuations_per_month` window=month.
  - `feature.guard:instant_push` on admin push routes consumes `instant_push_per_day` window=day.
  - Ensure route uses correct key for Saved Ads list: `view_deals` (currently `deal_view` in routes/api.php:26).
- To test quotas quickly:
  - Seed a user with active subscription using `free` plan (PlanEntitlementsSeeder).
  - Repeatedly hit guarded endpoints and inspect `user_quotas` table.

## Troubleshooting
- If `php artisan` fails: composer or vendor autoload missing — add composer.json and install dependencies.
- Queue not processing: ensure queue driver matches `.env` and worker running; check failed jobs table if configured.
- Telegraphed webhook 401: verify `TELEGRAPH_SECRET` and secure header matching.
- CTR not changing: ensure `push_logs.sent_at` and `delivered_at` are set by sender logic and use `/api/simulate/click` to mark clicks.
- KPIs missing: instrument EventBus emits and persist AI chats to `chats` table.
