# Testing Framework Documentation

This document provides comprehensive information about the testing framework included with the Laravel Bot Template.

## Overview

The testing framework is designed to ensure the reliability and quality of your Telegram bot modules. It includes unit tests, feature tests, integration tests, and example modules to help you build robust bot functionality.

## Test Structure

```
tests/
├── Feature/            # Feature tests for complete workflows
│   ├── Modules/        # Module-specific feature tests
│   └── ExampleTest.php # Basic feature test example
├── Unit/               # Unit tests for individual components
│   └── Services/       # Service layer unit tests
├── Integration/        # Integration tests for module interactions
├── Security/           # Security-related tests
└── TestCase.php       # Base test case with helper methods
```

## Available Tests

### Core Framework Tests

#### ModuleManagerTest
Tests the core module management functionality:
- Module registration and retrieval
- Command routing to appropriate modules
- Module dependency resolution
- Module state management (enable/disable)
- Duplicate module handling

**Key Test Methods:**
```php
test_can_register_module()
test_can_get_registered_modules()
test_can_find_module_for_command()
test_can_enable_and_disable_modules()
test_cannot_register_duplicate_module()
```

#### EventBusTest
Tests the event-driven communication system:
- Event listener registration
- Event dispatching and handling
- Multiple listeners for single events
- Event data passing
- Error handling in listeners

**Key Test Methods:**
```php
test_can_register_listener()
test_can_register_multiple_listeners()
test_listeners_receive_correct_data()
test_listener_exceptions_are_handled()
test_event_chaining()
```

#### ConfigurationManagerTest
Tests the configuration management system:
- Setting and getting configuration values
- Nested configuration handling
- Configuration merging
- File-based configuration loading/saving
- Configuration scoping

**Key Test Methods:**
```php
test_can_set_and_get_configuration()
test_can_work_with_nested_arrays()
test_can_merge_configurations()
test_can_load_from_file()
test_configuration_scoping()
```

### Module Tests

#### PersianModuleTest
Tests Persian localization functionality:
- Language command handling
- Language switching callbacks
- Persian number formatting
- Persian date formatting
- Localization services

**Key Test Methods:**
```php
test_handles_language_command()
test_handles_persian_callback()
test_provides_localization_service()
test_format_persian_numbers()
test_language_persistence()
```

#### PaymentModuleTest
Tests payment processing functionality:
- Payment command handling
- Zarinpal integration
- Payment verification
- Custom amount processing
- Payment history tracking

**Key Test Methods:**
```php
test_handles_payment_command()
test_handles_payment_amount_callback()
test_handles_payment_verification()
test_custom_amount_processing()
test_payment_limits()
```

#### ContentModuleTest
Tests content management functionality:
- Help and about commands
- Dynamic menu generation
- Content rendering with variables
- Multilingual content support
- Different content types (text, image, video)

**Key Test Methods:**
```php
test_handles_help_command()
test_dynamic_content_rendering()
test_multilingual_content()
test_keyboard_menu_generation()
test_content_cache()
```

#### AnalyticsModuleTest
Tests analytics and tracking functionality:
- Event tracking
- User statistics generation
- System analytics
- Admin access control
- Data aggregation

**Key Test Methods:**
```php
test_tracks_command_usage()
test_generates_user_statistics()
test_generates_system_analytics()
test_calculates_user_retention()
test_tracks_popular_commands()
```

#### NotificationModuleTest
Tests notification system functionality:
- Template management
- Broadcast campaigns
- User targeting
- Delivery tracking
- Notification scheduling

**Key Test Methods:**
```php
test_handles_notifications_command()
test_creates_notification_template()
test_sends_broadcast_notification()
test_tracks_notification_delivery()
test_handles_notification_preferences()
```

### Integration Tests

#### ModularSystemIntegrationTest
Tests how all modules work together:
- Module system initialization
- Inter-module communication via EventBus
- Command routing across modules
- Configuration sharing between modules
- Real-world user interaction flows

**Key Test Methods:**
```php
test_module_system_initialization()
test_inter_module_communication_via_eventbus()
test_command_routing_across_modules()
test_real_world_user_interaction_flow()
test_module_error_isolation()
```

## Running Tests

### Prerequisites

1. Ensure your testing environment is set up:
```bash
# Copy environment file for testing
cp .env.example .env.testing

# Set up test database
php artisan config:cache --env=testing
```

2. Install dependencies:
```bash
composer install
npm install
```

### Running All Tests

```bash
# Run all tests
php artisan test

# Run with coverage report
php artisan test --coverage

# Run tests in parallel (faster)
php artisan test --parallel
```

### Running Specific Test Types

```bash
# Run only unit tests
php artisan test --testsuite=Unit

# Run only feature tests
php artisan test --testsuite=Feature

# Run specific test file
php artisan test tests/Unit/Services/ModuleManagerTest.php

# Run specific test method
php artisan test --filter=test_can_register_module
```

### Running Tests by Group

```bash
# Run core framework tests
php artisan test tests/Unit/Services/

# Run module tests
php artisan test tests/Feature/Modules/

# Run integration tests
php artisan test tests/Integration/
```

## Writing New Tests

### Test Structure Guidelines

1. **Arrange, Act, Assert**: Structure your tests clearly
2. **Single Responsibility**: Each test should verify one specific behavior
3. **Descriptive Names**: Use clear, descriptive test method names
4. **Isolation**: Tests should not depend on each other

### Example Unit Test

```php
<?php

namespace Tests\Unit\Services;

use Tests\TestCase;
use App\Services\YourService;
use Mockery;

class YourServiceTest extends TestCase
{
    private YourService $service;
    
    protected function setUp(): void
    {
        parent::setUp();
        $this->service = new YourService();
    }
    
    protected function tearDown(): void
    {
        Mockery::close();
        parent::tearDown();
    }
    
    public function test_your_functionality()
    {
        // Arrange
        $input = 'test input';
        $expected = 'expected output';
        
        // Act
        $result = $this->service->processInput($input);
        
        // Assert
        $this->assertEquals($expected, $result);
    }
}
```

### Example Feature Test

```php
<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;

class YourFeatureTest extends TestCase
{
    use RefreshDatabase;
    
    public function test_user_can_complete_workflow()
    {
        // Arrange
        $user = User::factory()->create();
        
        // Act
        $response = $this->actingAs($user)
            ->post('/api/telegram/webhook', [
                'message' => [
                    'text' => '/start',
                    'from' => ['id' => $user->telegram_id],
                    'chat' => ['id' => 12345]
                ]
            ]);
        
        // Assert
        $response->assertStatus(200);
        $this->assertDatabaseHas('users', [
            'telegram_id' => $user->telegram_id,
            'last_activity' => now()
        ]);
    }
}
```

## Testing Best Practices

### 1. Use Database Transactions
```php
use Illuminate\Foundation\Testing\RefreshDatabase;

class YourTest extends TestCase
{
    use RefreshDatabase; // Automatically rolls back database changes
}
```

### 2. Mock External Services
```php
// Mock Telegram API calls
$this->mock(TelegramApiService::class, function ($mock) {
    $mock->shouldReceive('sendMessage')
         ->once()
         ->andReturn(['ok' => true]);
});
```

### 3. Use Factories for Test Data
```php
// Create test users with factories
$user = User::factory()->create(['telegram_id' => 12345]);
$users = User::factory()->count(10)->create();
```

### 4. Test Edge Cases
```php
public function test_handles_invalid_input()
{
    $result = $this->service->processInput('');
    $this->assertNull($result);
}

public function test_handles_extremely_long_input()
{
    $longInput = str_repeat('a', 10000);
    $result = $this->service->processInput($longInput);
    $this->assertIsString($result);
}
```

### 5. Use Data Providers for Multiple Scenarios
```php
/**
 * @dataProvider inputProvider
 */
public function test_processes_various_inputs($input, $expected)
{
    $result = $this->service->processInput($input);
    $this->assertEquals($expected, $result);
}

public function inputProvider()
{
    return [
        ['hello', 'HELLO'],
        ['world', 'WORLD'],
        ['123', '123'],
    ];
}
```

## Continuous Integration

### GitHub Actions Example

```yaml
name: Tests

on: [push, pull_request]

jobs:
  tests:
    runs-on: ubuntu-latest
    
    services:
      mysql:
        image: mysql:8.0
        env:
          MYSQL_ROOT_PASSWORD: password
          MYSQL_DATABASE: larbot_test
        options: >-
          --health-cmd="mysqladmin ping"
          --health-interval=10s
          --health-timeout=5s
          --health-retries=3

    steps:
    - uses: actions/checkout@v2
    
    - name: Setup PHP
      uses: shivammathur/setup-php@v2
      with:
        php-version: '8.3'
        extensions: mbstring, xml, ctype, iconv, intl, pdo_mysql
        
    - name: Install dependencies
      run: composer install --prefer-dist --no-progress
      
    - name: Copy environment file
      run: cp .env.testing .env
      
    - name: Generate application key
      run: php artisan key:generate
      
    - name: Run migrations
      run: php artisan migrate
      
    - name: Run tests
      run: php artisan test --coverage
```

## Performance Testing

### Database Query Optimization
```php
public function test_queries_are_optimized()
{
    DB::enableQueryLog();
    
    $this->service->getDataForUser(123);
    
    $queries = DB::getQueryLog();
    $this->assertLessThan(5, count($queries)); // Ensure we don't have N+1 queries
}
```

### Memory Usage Testing
```php
public function test_memory_usage_is_reasonable()
{
    $startMemory = memory_get_usage();
    
    $this->service->processBulkData($largeDataset);
    
    $endMemory = memory_get_usage();
    $memoryUsed = $endMemory - $startMemory;
    
    $this->assertLessThan(50 * 1024 * 1024, $memoryUsed); // Less than 50MB
}
```

## Test Data Management

### Using Factories
```php
// Define in database/factories/UserFactory.php
class UserFactory extends Factory
{
    public function definition()
    {
        return [
            'telegram_id' => $this->faker->unique()->numberBetween(100000, 999999),
            'first_name' => $this->faker->firstName,
            'username' => $this->faker->userName,
            'balance' => $this->faker->numberBetween(0, 100000),
        ];
    }
}

// Use in tests
$user = User::factory()->create();
$richUser = User::factory()->create(['balance' => 100000]);
$users = User::factory()->count(10)->create();
```

### Using Seeders for Complex Test Data
```php
// Run specific seeder for tests
$this->seed(TestDataSeeder::class);
```

## Debugging Tests

### Enable Debug Mode
```php
// In your test method
$this->withoutExceptionHandling(); // Show full stack traces
```

### Dump Variables
```php
// Debug variables in tests
dump($variable);
dd($variable); // Dump and die

// Assert and dump on failure
$this->assertEquals($expected, $actual, 'Custom error message');
```

### Log Test Output
```php
// Log information during tests
Log::info('Test checkpoint', ['data' => $testData]);
```

## Troubleshooting

### Common Issues

1. **Database Connection Errors**
   - Ensure test database is configured in `.env.testing`
   - Check database service is running

2. **Memory Limit Errors**
   - Increase memory limit in `phpunit.xml`
   - Use database transactions instead of migrations

3. **Timeout Issues**
   - Reduce test dataset sizes
   - Mock external API calls

4. **Permission Errors**
   - Ensure storage directory is writable
   - Check file permissions for test files

### Performance Tips

1. **Use Database Transactions**: Faster than migrations
2. **Mock External Services**: Avoid real API calls
3. **Parallel Testing**: Run tests in parallel when possible
4. **Optimize Factories**: Create minimal required data

## Example Module Testing

See the `app/Modules/Examples/` directory for complete example modules with corresponding tests. These examples demonstrate:

- Basic module structure and testing
- Advanced features like state management
- Complex workflows with user interaction
- Event-driven architecture
- Caching and performance optimization

The example modules serve as templates for building and testing your own modules.