<?php

namespace Tests\Feature\Modules;

use Tests\TestCase;
use App\Modules\Notification\NotificationModule;
use App\Services\Notification\NotificationService;
use App\Models\User;
use App\Models\UserNotificationSetting;
use App\DTOs\TelegramMessage;
use App\DTOs\TelegramCallback;
use Illuminate\Foundation\Testing\RefreshDatabase;

class NotificationModuleTest extends TestCase
{
    use RefreshDatabase;

    private NotificationModule $module;
    private User $user;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->module = app(NotificationModule::class);
        $this->user = User::factory()->create([
            'telegram_id' => '123456789',
            'name' => 'Test User'
        ]);
    }

    /** @test */
    public function it_handles_notifications_command()
    {
        $message = new TelegramMessage(
            chatId: 123456789,
            userId: 123456789,
            text: '/notifications',
            message: []
        );

        $response = $this->module->handleMessage($message);

        $this->assertNotNull($response);
        $this->assertStringContains('تنظیمات اعلان‌ها', $response->text);
        $this->assertNotNull($response->keyboard);
    }

    /** @test */
    public function it_handles_subscribe_command()
    {
        $message = new TelegramMessage(
            chatId: 123456789,
            userId: 123456789,
            text: '/subscribe',
            message: []
        );

        $response = $this->module->handleMessage($message);

        $this->assertNotNull($response);
        $this->assertStringContains('عضو شدید', $response->text);
        
        // Check that user settings were updated
        $settings = UserNotificationSetting::where('user_id', $this->user->id)->first();
        $this->assertNotNull($settings);
        $this->assertNull($settings->opted_out_at);
    }

    /** @test */
    public function it_handles_unsubscribe_command()
    {
        $message = new TelegramMessage(
            chatId: 123456789,
            userId: 123456789,
            text: '/unsubscribe',
            message: []
        );

        $response = $this->module->handleMessage($message);

        $this->assertNotNull($response);
        $this->assertStringContains('خارج شدید', $response->text);
        
        // Check that user was opted out
        $settings = UserNotificationSetting::where('user_id', $this->user->id)->first();
        $this->assertNotNull($settings);
        $this->assertNotNull($settings->opted_out_at);
    }

    /** @test */
    public function it_handles_notification_toggle_callback()
    {
        // Create initial settings
        UserNotificationSetting::forUser($this->user);

        $callback = new TelegramCallback(
            chatId: 123456789,
            userId: 123456789,
            data: 'notification_toggle_broadcasts',
            callbackQueryId: 'test_query_id',
            callbackQuery: []
        );

        $response = $this->module->handleCallback($callback);

        $this->assertNotNull($response);
        $this->assertStringContains('تنظیمات اعلان‌ها', $response->text);
    }

    /** @test */
    public function it_handles_opt_out_callback()
    {
        UserNotificationSetting::forUser($this->user);

        $callback = new TelegramCallback(
            chatId: 123456789,
            userId: 123456789,
            data: 'notification_optout',
            callbackQueryId: 'test_query_id',
            callbackQuery: []
        );

        $response = $this->module->handleCallback($callback);

        $this->assertNotNull($response);
        
        // Verify user was opted out
        $settings = $this->user->fresh()->notificationSettings;
        $this->assertTrue($settings->isOptedOut());
    }

    /** @test */
    public function it_returns_null_for_unhandled_messages()
    {
        $message = new TelegramMessage(
            chatId: 123456789,
            userId: 123456789,
            text: '/unknown_command',
            message: []
        );

        $response = $this->module->handleMessage($message);

        $this->assertNull($response);
    }

    /** @test */
    public function it_returns_null_for_unhandled_callbacks()
    {
        $callback = new TelegramCallback(
            chatId: 123456789,
            userId: 123456789,
            data: 'unknown_callback',
            callbackQueryId: 'test_query_id',
            callbackQuery: []
        );

        $response = $this->module->handleCallback($callback);

        $this->assertNull($response);
    }

    /** @test */
    public function it_has_correct_module_metadata()
    {
        $this->assertEquals('notification', $this->module->getModuleName());
        $this->assertEquals('1.0.0', $this->module->getModuleVersion());
        $this->assertEquals([], $this->module->getDependencies());
        $this->assertTrue($this->module->isEnabled());
        $this->assertEquals(50, $this->module->getPriority());
    }

    /** @test */
    public function it_provides_configuration_schema()
    {
        $schema = $this->module->getConfigurationSchema();

        $this->assertIsArray($schema);
        $this->assertArrayHasKey('enabled', $schema);
        $this->assertArrayHasKey('default_opt_in', $schema);
        $this->assertArrayHasKey('rate_limit_hours', $schema);
    }
}